<?php

namespace Drupal\simple_sitemap_xml\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\content_moderation\ModerationInformationInterface;

/**
 * Returns responses for Simple Sitemap XML routes.
 */
class SitemapController extends ControllerBase {

  /**
   * The menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface
   */
  protected $menuTree;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformationInterface|null
   */
  protected $moderationInformation;

  /**
   * Constructs a SitemapController object.
   *
   * @param \Drupal\Core\Menu\MenuLinkTreeInterface $menu_tree
   *   The menu link tree service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\content_moderation\ModerationInformationInterface|null $moderation_information
   *   The moderation information service (optional).
   */
  public function __construct(MenuLinkTreeInterface $menu_tree, RequestStack $request_stack, CacheBackendInterface $cache, EntityTypeManagerInterface $entity_type_manager, ModerationInformationInterface $moderation_information = NULL) {
    $this->menuTree = $menu_tree;
    $this->requestStack = $request_stack;
    $this->cache = $cache;
    $this->entityTypeManager = $entity_type_manager;
    $this->moderationInformation = $moderation_information;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $moderation_information = NULL;
    if ($container->has('content_moderation.moderation_information')) {
      $moderation_information = $container->get('content_moderation.moderation_information');
    }
    
    return new static(
      $container->get('menu.link_tree'),
      $container->get('request_stack'),
      $container->get('cache.default'),
      $container->get('entity_type.manager'),
      $moderation_information
    );
  }

  /**
   * Builds the sitemap.xml response.
   */
  public function sitemap() {
    $config = $this->config('simple_sitemap_xml.settings');
    
    // Get generation mode.
    $generation_mode = $config->get('generation_mode') ?? 'content_types';
    
    // Check cache first.
    $cache_key = 'simple_sitemap_xml:sitemap:' . $generation_mode;
    $cached = $this->cache->get($cache_key);
    if ($cached) {
      $response = new Response($cached->data);
      $response->headers->set('Content-Type', 'application/xml; charset=utf-8');
      return $response;
    }

    // Get excluded URLs.
    $excluded_urls = $this->getExcludedUrls($config->get('excluded_urls'));
    
    // Get base URL.
    $base_url = $config->get('base_url') ?: $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    
    // Get other settings.
    $priority = $config->get('priority') ?: '0.5';
    $changefreq = $config->get('changefreq') ?: 'weekly';
    $include_homepage = $config->get('include_homepage') ?? TRUE;

    $urls_with_levels = [];

    if ($generation_mode === 'content_types') {
      // Generate URLs from content types.
      $urls_with_levels = $this->generateUrlsFromContentTypes($base_url, $excluded_urls, $config);
    }
    else {
      // Legacy menu-based generation.
      $menu_name = $config->get('menu_selection');
      if (!$menu_name) {
        // Return empty sitemap if no menu is configured.
        return $this->generateEmptySitemap();
      }

      // Load menu tree.
      $parameters = new MenuTreeParameters();
      $parameters->setMaxDepth(10); // Set a reasonable depth limit.
      
      try {
        $tree = $this->menuTree->load($menu_name, $parameters);
      }
      catch (\Exception $e) {
        // If menu doesn't exist or has issues, return empty sitemap.
        return $this->generateEmptySitemap();
      }
      
      // Transform menu tree to get enabled links.
      $manipulators = [
        ['callable' => 'menu.default_tree_manipulators:checkAccess'],
        ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
      ];
      $tree = $this->menuTree->transform($tree, $manipulators);

      // Generate sitemap URLs.
      $urls_with_levels = $this->extractUrlsFromTree($tree, $base_url, $excluded_urls);
    }

    // Add homepage if enabled and not excluded.
    if ($include_homepage) {
      $homepage_url = $base_url . '/';
      if (!$this->isUrlExcluded('/', $homepage_url, $excluded_urls)) {
        // Homepage is level 0 (top level).
        array_unshift($urls_with_levels, ['url' => $homepage_url, 'level' => 0, 'content_type' => 'homepage']);
      }
    }

    // Get priority configuration.
    $priority_mode = $config->get('priority_mode') ?? 'default';
    $menu_level_priorities = $this->getMenuLevelPriorities($config->get('menu_level_priorities'));
    $custom_priorities = $this->getCustomPriorities($config->get('custom_priorities'));
    $content_type_priorities = $this->getContentTypePriorities($config->get('content_type_priorities'));

    // Generate XML.
    $xml = $this->generateSitemapXml($urls_with_levels, $priority, $changefreq, $priority_mode, $menu_level_priorities, $custom_priorities, $content_type_priorities);

    // Cache for 1 hour.
    $this->cache->set($cache_key, $xml, time() + 3600, ['simple_sitemap_xml']);

    // Create response.
    $response = new Response($xml);
    $response->headers->set('Content-Type', 'application/xml; charset=utf-8');
    
    return $response;
  }

  /**
   * Generate URLs from content types.
   *
   * @param string $base_url
   *   The base URL.
   * @param array $excluded_urls
   *   Array of excluded URLs.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   The module configuration.
   *
   * @return array
   *   Array of URLs with additional information.
   */
  protected function generateUrlsFromContentTypes($base_url, array $excluded_urls, $config) {
    $urls = [];
    
    // Get selected content types.
    $selected_content_types = $config->get('selected_content_types') ?? [];
    if (empty($selected_content_types)) {
      return $urls;
    }
    
    // Get node status filter.
    $status_filter = $config->get('node_status_filter') ?? ['published'];
    $include_published = in_array('published', $status_filter);
    $include_unpublished = in_array('unpublished', $status_filter);
    
    // Get node storage.
    $node_storage = $this->entityTypeManager->getStorage('node');
    
    foreach ($selected_content_types as $content_type) {
      if (empty($content_type)) {
        continue;
      }
      
      // Build query for this content type.
      $query = $node_storage->getQuery()
        ->condition('type', $content_type)
        ->accessCheck(TRUE)
        ->sort('nid');
      
      // Apply status filter.
      if ($include_published && $include_unpublished) {
        // Include both published and unpublished.
      }
      elseif ($include_published) {
        $query->condition('status', 1);
      }
      elseif ($include_unpublished) {
        $query->condition('status', 0);
      }
      else {
        // No status selected, skip this content type.
        continue;
      }
      
      $nids = $query->execute();
      
      if (empty($nids)) {
        continue;
      }
      
      // Load nodes and generate URLs.
      $nodes = $node_storage->loadMultiple($nids);
      
      foreach ($nodes as $node) {
        if ($node instanceof NodeInterface) {
          try {
            // Get node URL.
            $url_object = $node->toUrl('canonical', ['absolute' => FALSE]);
            $relative_url = $url_object->toString();
            $full_url = $base_url . $relative_url;
            
            // Check if URL should be excluded.
            if (!$this->isUrlExcluded($relative_url, $full_url, $excluded_urls)) {
              // Check moderation state if Content Moderation is enabled.
              if ($this->shouldIncludeNodeInSitemap($node, $include_published, $include_unpublished)) {
                $urls[] = [
                  'url' => $full_url,
                  'level' => 1, // All content nodes are level 1
                  'content_type' => $content_type,
                  'nid' => $node->id(),
                  'changed' => $node->getChangedTime(),
                ];
              }
            }
          }
          catch (\Exception $e) {
            // Skip nodes that can't generate URLs.
            continue;
          }
        }
      }
    }
    
    return $urls;
  }

  /**
   * Extract URLs from menu tree.
   *
   * @param array $tree
   *   The menu tree.
   * @param string $base_url
   *   The base URL.
   * @param array $excluded_urls
   *   Array of excluded URLs.
   * @param int $level
   *   Current menu level.
   *
   * @return array
   *   Array of URLs with levels.
   */
  protected function extractUrlsFromTree(array $tree, $base_url, array $excluded_urls, $level = 1) {
    $urls = [];

    foreach ($tree as $element) {
      if ($element->link->isEnabled()) {
        $url_object = $element->link->getUrlObject();
        
        // Only process internal URLs.
        if ($url_object->isRouted()) {
          try {
            $url = $url_object->toString();
            $full_url = $base_url . $url;
            
            // Check if URL should be excluded.
            if (!$this->isUrlExcluded($url, $full_url, $excluded_urls)) {
              $urls[] = ['url' => $full_url, 'level' => $level];
            }
          }
          catch (\Exception $e) {
            // Skip URLs that can't be generated.
            continue;
          }
        }
      }

      // Recursively process subtree.
      if ($element->subtree) {
        $subtree_urls = $this->extractUrlsFromTree($element->subtree, $base_url, $excluded_urls, $level + 1);
        $urls = array_merge($urls, $subtree_urls);
      }
    }

    return $urls;
  }

  /**
   * Check if a URL should be excluded.
   *
   * @param string $relative_url
   *   The relative URL.
   * @param string $full_url
   *   The full URL.
   * @param array $excluded_urls
   *   Array of excluded URLs.
   *
   * @return bool
   *   TRUE if URL should be excluded.
   */
  protected function isUrlExcluded($relative_url, $full_url, array $excluded_urls) {
    foreach ($excluded_urls as $excluded) {
      // Check both relative and full URL.
      if ($relative_url === $excluded || $full_url === $excluded) {
        return TRUE;
      }
      
      // Check if the excluded URL is a pattern (contains wildcards).
      if (strpos($excluded, '*') !== FALSE) {
        $pattern = '/^' . str_replace('*', '.*', preg_quote($excluded, '/')) . '$/';
        if (preg_match($pattern, $relative_url) || preg_match($pattern, $full_url)) {
          return TRUE;
        }
      }
    }
    
    return FALSE;
  }

  /**
   * Get excluded URLs as array.
   *
   * @param string $excluded_urls_text
   *   The excluded URLs text from config.
   *
   * @return array
   *   Array of excluded URLs.
   */
  protected function getExcludedUrls($excluded_urls_text) {
    if (empty($excluded_urls_text)) {
      return [];
    }

    $urls = explode("\n", $excluded_urls_text);
    $urls = array_map('trim', $urls);
    $urls = array_filter($urls); // Remove empty lines.
    
    return $urls;
  }

  /**
   * Get menu level priorities as array.
   *
   * @param string $menu_level_priorities_text
   *   The menu level priorities text from config.
   *
   * @return array
   *   Array of priorities keyed by level.
   */
  protected function getMenuLevelPriorities($menu_level_priorities_text) {
    if (empty($menu_level_priorities_text)) {
      // Default priorities by level.
      return [
        1 => '1.0',
        2 => '0.8',
        3 => '0.6',
        4 => '0.4',
      ];
    }

    $priorities = [];
    $lines = explode("\n", $menu_level_priorities_text);
    
    foreach ($lines as $line) {
      $line = trim($line);
      if (empty($line)) {
        continue;
      }
      
      $parts = explode('|', $line);
      if (count($parts) === 2) {
        $level = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate values.
        if (is_numeric($level) && is_numeric($priority) && $level >= 1 && $level <= 10 && $priority >= 0 && $priority <= 1) {
          $priorities[(int)$level] = $priority;
        }
      }
    }
    
    return $priorities;
  }

  /**
   * Get content type priorities as array.
   *
   * @param string $content_type_priorities_text
   *   The content type priorities text from config.
   *
   * @return array
   *   Array of priorities keyed by content type.
   */
  protected function getContentTypePriorities($content_type_priorities_text) {
    if (empty($content_type_priorities_text)) {
      return [];
    }

    $priorities = [];
    $lines = explode("\n", $content_type_priorities_text);
    
    foreach ($lines as $line) {
      $line = trim($line);
      if (empty($line)) {
        continue;
      }
      
      $parts = explode('|', $line);
      if (count($parts) === 2) {
        $content_type = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate priority value.
        if (is_numeric($priority) && $priority >= 0 && $priority <= 1) {
          $priorities[$content_type] = $priority;
        }
      }
    }
    
    return $priorities;
  }

  /**
   * Get custom priorities as array.
   *
   * @param string $custom_priorities_text
   *   The custom priorities text from config.
   *
   * @return array
   *   Array of custom priorities keyed by URL.
   */
  protected function getCustomPriorities($custom_priorities_text) {
    if (empty($custom_priorities_text)) {
      return [];
    }

    $priorities = [];
    $lines = explode("\n", $custom_priorities_text);
    
    foreach ($lines as $line) {
      $line = trim($line);
      if (empty($line)) {
        continue;
      }
      
      $parts = explode('|', $line);
      if (count($parts) === 2) {
        $url = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate priority value.
        if (is_numeric($priority) && $priority >= 0 && $priority <= 1) {
          $priorities[$url] = $priority;
        }
      }
    }
    
    return $priorities;
  }

  /**
   * Generate sitemap XML.
   *
   * @param array $urls_with_levels
   *   Array of URLs with level information.
   * @param string $default_priority
   *   Default priority.
   * @param string $changefreq
   *   Default change frequency.
   * @param string $priority_mode
   *   Priority assignment mode.
   * @param array $menu_level_priorities
   *   Array of priorities by menu level.
   * @param array $custom_priorities
   *   Array of custom priorities keyed by URL.
   * @param array $content_type_priorities
   *   Array of priorities keyed by content type.
   *
   * @return string
   *   The XML content.
   */
  protected function generateSitemapXml(array $urls_with_levels, $default_priority, $changefreq, $priority_mode, array $menu_level_priorities = [], array $custom_priorities = [], array $content_type_priorities = []) {
    // Get the base URL for the XSL stylesheet.
    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $stylesheet_url = $base_url . '/sitemap.xsl';
    
    $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<?xml-stylesheet type="text/xsl" href="' . $stylesheet_url . '"?>' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

    $current_date = new DrupalDateTime();
    $lastmod = $current_date->format('Y-m-d\TH:i:s\Z');

    // Remove duplicates by URL.
    $unique_urls = [];
    foreach ($urls_with_levels as $item) {
      $url = $item['url'];
      if (!isset($unique_urls[$url])) {
        $unique_urls[$url] = $item;
      }
    }

    foreach ($unique_urls as $item) {
      $url = $item['url'];
      $level = $item['level'];
      $content_type = $item['content_type'] ?? null;
      $changed_time = $item['changed'] ?? null;
      
      // Get priority for this URL.
      $priority = $this->getPriorityForUrl($url, $level, $default_priority, $priority_mode, $menu_level_priorities, $custom_priorities, $content_type, $content_type_priorities);
      
      // Use node's changed time if available, otherwise use current time.
      $item_lastmod = $lastmod;
      if ($changed_time) {
        $changed_date = new DrupalDateTime();
        $changed_date->setTimestamp($changed_time);
        $item_lastmod = $changed_date->format('Y-m-d\TH:i:s\Z');
      }
      
      $xml .= '  <url>' . "\n";
      $xml .= '    <loc>' . htmlspecialchars($url, ENT_XML1, 'UTF-8') . '</loc>' . "\n";
      $xml .= '    <lastmod>' . $item_lastmod . '</lastmod>' . "\n";
      $xml .= '    <changefreq>' . $changefreq . '</changefreq>' . "\n";
      $xml .= '    <priority>' . $priority . '</priority>' . "\n";
      $xml .= '  </url>' . "\n";
    }

    $xml .= '</urlset>';

    return $xml;
  }

  /**
   * Get priority for a specific URL.
   *
   * @param string $url
   *   The full URL.
   * @param int $level
   *   The menu level.
   * @param string $default_priority
   *   The default priority.
   * @param string $priority_mode
   *   The priority assignment mode.
   * @param array $menu_level_priorities
   *   Array of priorities by menu level.
   * @param array $custom_priorities
   *   Array of custom priorities.
   * @param string $content_type
   *   The content type (if applicable).
   * @param array $content_type_priorities
   *   Array of priorities by content type.
   *
   * @return string
   *   The priority for this URL.
   */
  protected function getPriorityForUrl($url, $level, $default_priority, $priority_mode, array $menu_level_priorities, array $custom_priorities, $content_type = null, array $content_type_priorities = []) {
    // Parse URL to get path.
    $parsed = parse_url($url);
    $path = $parsed['path'] ?? '/';
    
    switch ($priority_mode) {
      case 'custom':
        return $this->getCustomPriorityForUrl($url, $path, $custom_priorities, $default_priority);
        
      case 'menu_level':
        return $this->getMenuLevelPriorityForUrl($level, $menu_level_priorities, $default_priority);
        
      case 'mixed':
        // Homepage always gets highest priority.
        if ($content_type === 'homepage') {
          return '1.0';
        }
        
        // First try custom priorities, then content type, then menu level, then default.
        $custom_priority = $this->getCustomPriorityForUrl($url, $path, $custom_priorities, null);
        if ($custom_priority !== null) {
          return $custom_priority;
        }
        
        // Try content type priority.
        if ($content_type && isset($content_type_priorities[$content_type])) {
          return $content_type_priorities[$content_type];
        }
        
        return $this->getMenuLevelPriorityForUrl($level, $menu_level_priorities, $default_priority);
        
      case 'content_type':
        // Homepage always gets highest priority.
        if ($content_type === 'homepage') {
          return '1.0';
        }
        // Use content type priority if available.
        if ($content_type && isset($content_type_priorities[$content_type])) {
          return $content_type_priorities[$content_type];
        }
        return $default_priority;
        
      case 'default':
      default:
        return $default_priority;
    }
  }

  /**
   * Get custom priority for a URL.
   *
   * @param string $url
   *   The full URL.
   * @param string $path
   *   The URL path.
   * @param array $custom_priorities
   *   Array of custom priorities.
   * @param string $fallback
   *   Fallback priority.
   *
   * @return string|null
   *   The priority or null if not found.
   */
  protected function getCustomPriorityForUrl($url, $path, array $custom_priorities, $fallback) {
    // Check exact matches first.
    if (isset($custom_priorities[$url])) {
      return $custom_priorities[$url];
    }
    
    if (isset($custom_priorities[$path])) {
      return $custom_priorities[$path];
    }
    
    // Check pattern matches.
    foreach ($custom_priorities as $pattern => $priority) {
      if (strpos($pattern, '*') !== FALSE) {
        $regex_pattern = '/^' . str_replace('*', '.*', preg_quote($pattern, '/')) . '$/';
        if (preg_match($regex_pattern, $url) || preg_match($regex_pattern, $path)) {
          return $priority;
        }
      }
    }
    
    return $fallback;
  }

  /**
   * Get menu level priority for a URL.
   *
   * @param int $level
   *   The menu level.
   * @param array $menu_level_priorities
   *   Array of priorities by level.
   * @param string $default_priority
   *   Default priority.
   *
   * @return string
   *   The priority.
   */
  protected function getMenuLevelPriorityForUrl($level, array $menu_level_priorities, $default_priority) {
    if (isset($menu_level_priorities[$level])) {
      return $menu_level_priorities[$level];
    }
    
    // For homepage (level 0), use highest priority if available.
    if ($level === 0 && isset($menu_level_priorities[1])) {
      return $menu_level_priorities[1];
    }
    
    return $default_priority;
  }

  /**
   * Generate empty sitemap.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Empty sitemap response.
   */
  protected function generateEmptySitemap() {
    $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
    $xml .= '</urlset>';

    $response = new Response($xml);
    $response->headers->set('Content-Type', 'application/xml; charset=utf-8');
    
    return $response;
  }

  /**
   * Serves the XSL stylesheet for the sitemap.
   */
  public function stylesheet() {
    $module_path = \Drupal::service('extension.list.module')->getPath('simple_sitemap_xml');
    $xsl_file = DRUPAL_ROOT . '/' . $module_path . '/sitemap.xsl';
    
    if (file_exists($xsl_file)) {
      $content = file_get_contents($xsl_file);
      $response = new Response($content);
      $response->headers->set('Content-Type', 'application/xml; charset=utf-8');
      return $response;
    }
    
    // Return empty response if file doesn't exist.
    $response = new Response('');
    $response->headers->set('Content-Type', 'application/xml; charset=utf-8');
    return $response;
  }

  /**
   * Determines if a node should be included in the sitemap.
   *
   * This method checks both the traditional status field and the moderation
   * state when Content Moderation is enabled.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check.
   * @param bool $include_published
   *   Whether to include published nodes.
   * @param bool $include_unpublished
   *   Whether to include unpublished nodes.
   *
   * @return bool
   *   TRUE if the node should be included in the sitemap.
   */
  protected function shouldIncludeNodeInSitemap(NodeInterface $node, $include_published, $include_unpublished) {
    // If Content Moderation is not available, fall back to status field.
    if (!$this->moderationInformation || !$this->moderationInformation->isModeratedEntity($node)) {
      $is_published = $node->isPublished();
      return ($include_published && $is_published) || ($include_unpublished && !$is_published);
    }

    // Content Moderation is enabled for this node type.
    // Check if the current revision is the live (published) revision.
    $is_live_revision = $this->moderationInformation->isLiveRevision($node);
    
    if ($is_live_revision) {
      // This is the live revision, include if published nodes are requested.
      return $include_published;
    }
    else {
      // This is not the live revision (draft, pending review, etc.)
      // Include only if unpublished nodes are specifically requested.
      return $include_unpublished;
    }
  }

}
