<?php

namespace Drupal\simple_sitemap_xml\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\system\Entity\Menu;
use Drupal\node\Entity\NodeType;

/**
 * Configure Simple Sitemap XML settings for this site.
 */
class SimpleSitemapXmlConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   *
   * @var string
   */
  const SETTINGS = 'simple_sitemap_xml.settings';

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simple_sitemap_xml_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);

    // Get all available content types.
    $content_types = NodeType::loadMultiple();
    $content_type_options = [];
    foreach ($content_types as $type_id => $type) {
      $content_type_options[$type_id] = $type->label();
    }

    $form['generation_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Sitemap generation mode'),
      '#options' => [
        'content_types' => $this->t('Generate from content types'),
        'menu' => $this->t('Generate from menu (legacy)'),
      ],
      '#default_value' => $config->get('generation_mode') ?? 'content_types',
      '#required' => TRUE,
    ];

    $form['content_type_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Type Settings'),
      '#states' => [
        'visible' => [
          ':input[name="generation_mode"]' => ['value' => 'content_types'],
        ],
      ],
    ];

    $form['content_type_settings']['selected_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types to include'),
      '#description' => $this->t('Select which content types should be included in the sitemap.'),
      '#options' => $content_type_options,
      '#default_value' => $config->get('selected_content_types') ?? [],
    ];

    $form['content_type_settings']['content_type_priorities'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content type priorities'),
      '#description' => $this->t('Set custom priorities for each content type. Format: one per line as "content_type|priority". Example:<br>page|1.0<br>article|0.8<br>news|0.9'),
      '#default_value' => $config->get('content_type_priorities'),
      '#rows' => 8,
    ];

    $form['content_type_settings']['node_status_filter'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Node status filter'),
      '#description' => $this->t('Include only nodes with the selected status.'),
      '#options' => [
        'published' => $this->t('Published'),
        'unpublished' => $this->t('Unpublished'),
      ],
      '#default_value' => $config->get('node_status_filter') ?? ['published'],
    ];

    // Get all available menus.
    $menus = Menu::loadMultiple();
    $menu_options = [];
    foreach ($menus as $menu_id => $menu) {
      $menu_options[$menu_id] = $menu->label();
    }

    $form['menu_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Menu Settings (Legacy)'),
      '#states' => [
        'visible' => [
          ':input[name="generation_mode"]' => ['value' => 'menu'],
        ],
      ],
    ];

    $form['menu_settings']['menu_selection'] = [
      '#type' => 'select',
      '#title' => $this->t('Menu for sitemap generation'),
      '#description' => $this->t('Select the menu to use for generating the sitemap.xml file.'),
      '#options' => $menu_options,
      '#default_value' => $config->get('menu_selection'),
    ];

    $form['include_homepage'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include homepage'),
      '#description' => $this->t('Automatically include the homepage (/) in the sitemap.'),
      '#default_value' => $config->get('include_homepage') ?? TRUE,
    ];

    $form['excluded_urls'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Excluded URLs'),
      '#description' => $this->t('Enter URLs to exclude from the sitemap.xml, one per line. You can use relative paths (e.g., /admin) or full URLs. Wildcards (*) are supported for patterns like /admin/*'),
      '#default_value' => $config->get('excluded_urls'),
      '#rows' => 10,
    ];

    $form['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#description' => $this->t('Base URL to use for generating full URLs in the sitemap. Leave empty to use the current site base URL.'),
      '#default_value' => $config->get('base_url'),
    ];

    $form['priority_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Priority Settings'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    $form['priority_settings']['priority_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Priority assignment mode'),
      '#description' => $this->t('Choose how to assign priorities to URLs in the sitemap.'),
      '#options' => [
        'default' => $this->t('Default priority for all URLs'),
        'content_type' => $this->t('Priority based on content type'),
        'menu_level' => $this->t('Automatic priority based on menu level'),
        'custom' => $this->t('Custom priorities for specific URLs'),
        'mixed' => $this->t('Content type + custom overrides'),
      ],
      '#default_value' => $config->get('priority_mode') ?? 'content_type',
    ];

    $form['priority_settings']['priority'] = [
      '#type' => 'select',
      '#title' => $this->t('Default priority'),
      '#description' => $this->t('Default priority for pages in the sitemap (used when mode is "Default" or as fallback).'),
      '#options' => [
        '0.1' => '0.1',
        '0.2' => '0.2',
        '0.3' => '0.3',
        '0.4' => '0.4',
        '0.5' => '0.5',
        '0.6' => '0.6',
        '0.7' => '0.7',
        '0.8' => '0.8',
        '0.9' => '0.9',
        '1.0' => '1.0',
      ],
      '#default_value' => $config->get('priority') ?? '0.5',
    ];

    $form['priority_settings']['menu_level_priorities'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Menu level priorities'),
      '#description' => $this->t('Set priorities by menu level. Format: one per line as "level|priority". Example:<br>1|1.0<br>2|0.8<br>3|0.6<br>4|0.4'),
      '#default_value' => $config->get('menu_level_priorities') ?? "1|1.0\n2|0.8\n3|0.6\n4|0.4",
      '#rows' => 6,
      '#states' => [
        'visible' => [
          ':input[name="priority_mode"]' => [
            ['value' => 'menu_level'],
            ['value' => 'mixed'],
          ],
        ],
      ],
    ];

    $form['priority_settings']['content_type_priorities_settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content type priorities (for priority settings)'),
      '#description' => $this->t('Set custom priorities for each content type (also used in priority settings). Format: one per line as "content_type|priority". Example:<br>page|1.0<br>article|0.8<br>news|0.9'),
      '#default_value' => $config->get('content_type_priorities'),
      '#rows' => 8,
      '#states' => [
        'visible' => [
          ':input[name="priority_mode"]' => [
            ['value' => 'content_type'],
            ['value' => 'mixed'],
          ],
        ],
      ],
    ];

    $form['priority_settings']['custom_priorities'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom priorities'),
      '#description' => $this->t('Set custom priorities for specific URLs. Format: one per line as "URL|priority". Example:<br>/about|0.8<br>/contact|0.6<br>https://example.com/important|1.0'),
      '#default_value' => $config->get('custom_priorities'),
      '#rows' => 8,
      '#states' => [
        'visible' => [
          ':input[name="priority_mode"]' => [
            ['value' => 'custom'],
            ['value' => 'mixed'],
          ],
        ],
      ],
    ];

    $form['changefreq'] = [
      '#type' => 'select',
      '#title' => $this->t('Change frequency'),
      '#description' => $this->t('How frequently pages are updated.'),
      '#options' => [
        'always' => $this->t('Always'),
        'hourly' => $this->t('Hourly'),
        'daily' => $this->t('Daily'),
        'weekly' => $this->t('Weekly'),
        'monthly' => $this->t('Monthly'),
        'yearly' => $this->t('Yearly'),
        'never' => $this->t('Never'),
      ],
      '#default_value' => $config->get('changefreq') ?? 'weekly',
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save configuration'),
    ];

    $form['actions']['clear_cache'] = [
      '#type' => 'submit',
      '#value' => $this->t('Clear sitemap cache'),
      '#submit' => ['::clearCache'],
      '#limit_validation_errors' => [],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Validate content type priorities.
    $content_type_priorities = $form_state->getValue('content_type_priorities');
    if (!empty($content_type_priorities)) {
      $lines = explode("\n", $content_type_priorities);
      foreach ($lines as $line_number => $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }
        
        $parts = explode('|', $line);
        if (count($parts) !== 2) {
          $form_state->setErrorByName('content_type_priorities', $this->t('Line @line: Invalid format. Use "content_type|priority" format.', [
            '@line' => $line_number + 1,
          ]));
          continue;
        }
        
        $content_type = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate priority value.
        if (!is_numeric($priority) || $priority < 0 || $priority > 1) {
          $form_state->setErrorByName('content_type_priorities', $this->t('Line @line: Priority "@priority" must be a number between 0.0 and 1.0.', [
            '@line' => $line_number + 1,
            '@priority' => $priority,
          ]));
        }
      }
    }

    $excluded_urls = $form_state->getValue('excluded_urls');
    if (!empty($excluded_urls)) {
      $urls = explode("\n", $excluded_urls);
      foreach ($urls as $line_number => $url) {
        $url = trim($url);
        if (empty($url)) {
          continue;
        }
        
        // Basic validation - check if it looks like a URL or path.
        if (!preg_match('/^(https?:\/\/|\/|\*)/', $url)) {
          $form_state->setErrorByName('excluded_urls', $this->t('Line @line: "@url" does not appear to be a valid URL. Must start with http://, https://, / or *', [
            '@line' => $line_number + 1,
            '@url' => $url,
          ]));
        }
      }
    }

    // Validate menu level priorities.
    $menu_level_priorities = $form_state->getValue('menu_level_priorities');
    if (!empty($menu_level_priorities)) {
      $lines = explode("\n", $menu_level_priorities);
      foreach ($lines as $line_number => $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }
        
        $parts = explode('|', $line);
        if (count($parts) !== 2) {
          $form_state->setErrorByName('menu_level_priorities', $this->t('Line @line: Invalid format. Use "level|priority" format.', [
            '@line' => $line_number + 1,
          ]));
          continue;
        }
        
        $level = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate level.
        if (!is_numeric($level) || $level < 1 || $level > 10) {
          $form_state->setErrorByName('menu_level_priorities', $this->t('Line @line: Level "@level" must be a number between 1 and 10.', [
            '@line' => $line_number + 1,
            '@level' => $level,
          ]));
        }
        
        // Validate priority value.
        if (!is_numeric($priority) || $priority < 0 || $priority > 1) {
          $form_state->setErrorByName('menu_level_priorities', $this->t('Line @line: Priority "@priority" must be a number between 0.0 and 1.0.', [
            '@line' => $line_number + 1,
            '@priority' => $priority,
          ]));
        }
      }
    }

    // Validate custom priorities.
    $custom_priorities = $form_state->getValue('custom_priorities');
    if (!empty($custom_priorities)) {
      $lines = explode("\n", $custom_priorities);
      foreach ($lines as $line_number => $line) {
        $line = trim($line);
        if (empty($line)) {
          continue;
        }
        
        $parts = explode('|', $line);
        if (count($parts) !== 2) {
          $form_state->setErrorByName('custom_priorities', $this->t('Line @line: Invalid format. Use "URL|priority" format.', [
            '@line' => $line_number + 1,
          ]));
          continue;
        }
        
        $url = trim($parts[0]);
        $priority = trim($parts[1]);
        
        // Validate URL format.
        if (!preg_match('/^(https?:\/\/|\/|\*)/', $url)) {
          $form_state->setErrorByName('custom_priorities', $this->t('Line @line: "@url" does not appear to be a valid URL.', [
            '@line' => $line_number + 1,
            '@url' => $url,
          ]));
        }
        
        // Validate priority value.
        if (!is_numeric($priority) || $priority < 0 || $priority > 1) {
          $form_state->setErrorByName('custom_priorities', $this->t('Line @line: Priority "@priority" must be a number between 0.0 and 1.0.', [
            '@line' => $line_number + 1,
            '@priority' => $priority,
          ]));
        }
      }
    }
    
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Filter selected content types to remove empty values.
    $selected_content_types = array_filter($form_state->getValue('selected_content_types'));

    // Retrieve the configuration.
    $this->configFactory->getEditable(static::SETTINGS)
      ->set('generation_mode', $form_state->getValue('generation_mode'))
      ->set('selected_content_types', $selected_content_types)
      ->set('content_type_priorities', $form_state->getValue('content_type_priorities'))
      ->set('node_status_filter', array_filter($form_state->getValue('node_status_filter')))
      ->set('menu_selection', $form_state->getValue('menu_selection'))
      ->set('include_homepage', $form_state->getValue('include_homepage'))
      ->set('excluded_urls', $form_state->getValue('excluded_urls'))
      ->set('base_url', $form_state->getValue('base_url'))
      ->set('priority_mode', $form_state->getValue('priority_mode'))
      ->set('priority', $form_state->getValue('priority'))
      ->set('menu_level_priorities', $form_state->getValue('menu_level_priorities'))
      ->set('custom_priorities', $form_state->getValue('custom_priorities'))
      ->set('changefreq', $form_state->getValue('changefreq'))
      ->save();

    // Clear sitemap cache when configuration changes.
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_sitemap_xml']);

    $this->messenger()->addMessage($this->t('The configuration has been saved and the sitemap cache has been cleared.'));
  }

  /**
   * Clear sitemap cache.
   */
  public function clearCache(array &$form, FormStateInterface $form_state) {
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['simple_sitemap_xml']);
    $this->messenger()->addMessage($this->t('The sitemap cache has been cleared successfully.'));
  }

}
