<?php

namespace Drupal\simple_social_share\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Controller\TitleResolverInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a configurable social sharing block.
 *
 * @Block(
 *   id = "simple_social_share_block",
 *   admin_label = @Translation("Simple Social Share Block"),
 *   category = @Translation("Social")
 * )
 */
class SimpleSocialShareBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Available social media platforms.
   */
  protected const AVAILABLE_PLATFORMS = [
    'facebook' => 'Facebook',
    'twitter' => 'Twitter',
    'linkedin' => 'LinkedIn',
    'whatsapp' => 'WhatsApp',
    'telegram' => 'Telegram',
    'pinterest' => 'Pinterest',
    'reddit' => 'Reddit',
    'tumblr' => 'Tumblr',
    'email' => 'Email',
  ];

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The title resolver service.
   *
   * @var \Drupal\Core\Controller\TitleResolverInterface
   */
  protected $titleResolver;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new SimpleSocialShareBlock instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Drupal\Core\Controller\TitleResolverInterface $title_resolver
   *   The title resolver service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RouteMatchInterface $route_match, TitleResolverInterface $title_resolver, RequestStack $request_stack) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->routeMatch = $route_match;
    $this->titleResolver = $title_resolver;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('current_route_match'),
      $container->get('title_resolver'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $config = $this->getConfiguration();

    $form['platforms'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Social Media Platforms'),
      '#description' => $this->t('Select which social media platforms to display.'),
    ];

    foreach (self::AVAILABLE_PLATFORMS as $key => $label) {
      $form['platforms'][$key] = [
        '#type' => 'checkbox',
        '#title' => $this->t($label),
        '#default_value' => $config['platforms-' . $key],
      ];
    }

    $form['show_copy_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Copy Link Button'),
      '#description' => $this->t('Display a button to copy the current page URL.'),
      '#default_value' => $config['show_copy_link'] ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    foreach (self::AVAILABLE_PLATFORMS as $key => $label) {
      if ($form_state->getValue(['platforms', $key])) {
        $this->setConfigurationValue('platforms-' . $key, TRUE);
      }
      else {
        $this->setConfigurationValue('platforms-' . $key, FALSE);
      }
    }
    $this->setConfigurationValue('show_copy_link', $form_state->getValue('show_copy_link'));
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $route_name = $this->routeMatch->getRouteName();

    if ($route_name === 'entity.node.canonical') {
      $node = $this->routeMatch->getParameter('node');
      $current_url = Url::fromRoute('simple_social_share.node_short_link', ['node' => $node->id()])->setAbsolute()->toString();
    }
    elseif ($route_name === 'entity.taxonomy_term.canonical') {
      $term = $this->routeMatch->getParameter('taxonomy_term');
      $current_url = Url::fromRoute('simple_social_share.taxonomy_term_short_link', ['taxonomy_term' => $term->id()])->setAbsolute()->toString();
    }
    else {
      $current_url = Url::fromRoute('<current>')->setAbsolute()->toString();
    }

    $route_object = $this->routeMatch->getRouteObject();
    $current_title = '';
    if ($route_object) {
      $current_title = $this->titleResolver->getTitle($this->requestStack->getCurrentRequest(), $route_object);
    }
    // Ensure the title is always a string.
    $current_title = (string) $current_title;
    $config = $this->getConfiguration();
    // Build sharing URLs only for enabled platforms.
    $share_urls = [];
    $allowed_platforms = [];
    foreach (self::AVAILABLE_PLATFORMS as $key => $label) {
      if ($config['platforms-' . $key]) {
        $allowed_platforms[] = $key;
      }
    }
    foreach ($allowed_platforms as $platform) {
      switch ($platform) {
        case 'facebook':
          $share_urls['facebook'] = 'https://www.facebook.com/sharer/sharer.php?u=' . urlencode($current_url);
          break;

        case 'twitter':
          $share_urls['twitter'] = 'https://twitter.com/intent/tweet?url=' . urlencode($current_url) . '&text=' . urlencode($current_title);
          break;

        case 'linkedin':
          $share_urls['linkedin'] = 'https://www.linkedin.com/sharing/share-offsite/?url=' . urlencode($current_url);
          break;

        case 'whatsapp':
          $share_urls['whatsapp'] = 'https://wa.me/?text=' . urlencode($current_url);
          break;

        case 'telegram':
          $share_urls['telegram'] = 'https://t.me/share/url?url=' . urlencode($current_url) . '&text=' . urlencode($current_title);
          break;

        case 'pinterest':
          $share_urls['pinterest'] = 'https://pinterest.com/pin/create/button/?url=' . urlencode($current_url) . '&description=' . urlencode($current_title);
          break;

        case 'reddit':
          $share_urls['reddit'] = 'https://reddit.com/submit?url=' . urlencode($current_url) . '&title=' . urlencode($current_title);
          break;

        case 'tumblr':
          $share_urls['tumblr'] = 'https://www.tumblr.com/share/link?url=' . urlencode($current_url) . '&name=' . urlencode($current_title);
          break;

        case 'email':
          $share_urls['email'] = 'mailto:?subject=' . urlencode($current_title) . '&body=' . urlencode($current_url);
          break;
      }
    }

    return [
      '#theme' => 'simple_social_share_block',
      '#share_urls' => $share_urls,
      '#current_url' => $current_url,
      '#show_copy_link' => $config['show_copy_link'] ?? TRUE,
      '#attached' => [
        'library' => ['simple_social_share/social_share'],
      ],
    ];
  }

}
