/**
 * @file simple-toasts.js
 *
 * Replaces the standard drupal status messages and creates javascript toasts
 */
(function ($, Drupal, once, drupalSettings) {

  'use strict';

  Drupal.simpleToasts = {
    id: 'drupal-simple-toasts',
    context: '[data-drupal-messages]',
    fallbackContext: '[data-drupal-messages-fallback]',
    managedFileContext: '.js-form-managed-file',
    settingsInitialised: false,
    animation: {
      in: 'slide-in',
      out: 'slide-out',
      duration: 300,
    },
    messageTypeDuration: {
      status: 10000,
      warning: 10000,
      error: 0,
    },
    position: 'top-right', /** top-left/top-right/top-center/bottom-left/bottom-right/bottom-center **/
    style: 'default', /** theme **/
    iconStatus: '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" viewBox="0 0 16 16"><path d="M2.5 8a5.5 5.5 0 0 1 8.25-4.764.5.5 0 0 0 .5-.866A6.5 6.5 0 1 0 14.5 8a.5.5 0 0 0-1 0 5.5 5.5 0 1 1-11 0"/><path d="M15.354 3.354a.5.5 0 0 0-.708-.708L8 9.293 5.354 6.646a.5.5 0 1 0-.708.708l3 3a.5.5 0 0 0 .708 0z"/></svg>',
    iconWarning: '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" viewBox="0 0 16 16"><path d="M7.938 2.016A.13.13 0 0 1 8.002 2a.13.13 0 0 1 .063.016.15.15 0 0 1 .054.057l6.857 11.667c.036.06.035.124.002.183a.2.2 0 0 1-.054.06.1.1 0 0 1-.066.017H1.146a.1.1 0 0 1-.066-.017.2.2 0 0 1-.054-.06.18.18 0 0 1 .002-.183L7.884 2.073a.15.15 0 0 1 .054-.057m1.044-.45a1.13 1.13 0 0 0-1.96 0L.165 13.233c-.457.778.091 1.767.98 1.767h13.713c.889 0 1.438-.99.98-1.767z"/><path d="M7.002 12a1 1 0 1 1 2 0 1 1 0 0 1-2 0M7.1 5.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0z"/></svg>',
    iconError: '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" fill="currentColor" viewBox="0 0 16 16"><path d="M4.54.146A.5.5 0 0 1 4.893 0h6.214a.5.5 0 0 1 .353.146l4.394 4.394a.5.5 0 0 1 .146.353v6.214a.5.5 0 0 1-.146.353l-4.394 4.394a.5.5 0 0 1-.353.146H4.893a.5.5 0 0 1-.353-.146L.146 11.46A.5.5 0 0 1 0 11.107V4.893a.5.5 0 0 1 .146-.353zM5.1 1 1 5.1v5.8L5.1 15h5.8l4.1-4.1V5.1L10.9 1z"/><path d="M7.002 11a1 1 0 1 1 2 0 1 1 0 0 1-2 0M7.1 4.995a.905.905 0 1 1 1.8 0l-.35 3.507a.552.552 0 0 1-1.1 0z"/></svg>',
    iconClose: '<svg width="18" height="18" viewBox="0 0 44 44" aria-hidden="true" focusable="false"><path d="M0.549989 4.44999L4.44999 0.549988L43.45 39.55L39.55 43.45L0.549989 4.44999Z" /><path d="M39.55 0.549988L43.45 4.44999L4.44999 43.45L0.549988 39.55L39.55 0.549988Z" /></svg>',
    //iconClose: '<svg width="18" height="18" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg"><g fill="none" stroke="currentColor" stroke-width="2.25" stroke-linecap="round" stroke-linejoin="round"><path d="M20 4L4 20" /><path d="M4 4L20 20" /></g></svg>',
    count: 0,
    totalRemovingHeight: 0,
    messageWrappers: [], // A collection of message wrappers (if there are multiple)
    messageNodes: [], // The array of message nodes extracted from the DOM
    messages: [], // The array of message objects sent to the render function
    init: function (element) {
      this.messageWrappers = Array.from(element.children);
      if(!this.settingsInitialised) {
        this.getSettings();
      }
      this.messageObserver();
      this.run();
    },
    //If there is no message area, a fallback area for messages is created by Drupal for JS Drupal.Message() API
    initJsFallback: function (element) {
      const observer = new MutationObserver(mutationsList => {
        mutationsList.forEach(mutation => {
          this.init(mutation.target);
          observer.disconnect();
        });
      });
      observer.observe(element, {
        attributes: true,
        attributeFilter: ['data-drupal-messages'],
        childList: false,
        subtree: false,
      });
    },
    // Managed file inline form errors are a special case, we bypass most of the early initialisation and move straight to rendering
    initManagedFile: function (element) {
      const observer = new MutationObserver(mutationsList => {
        mutationsList.forEach(mutation => {
          if(mutation.type === 'childList' && mutation.addedNodes.length) {
            mutation.addedNodes.forEach(node => {
              if(node.nodeType === 1 && node.matches('.messages')) {
                //Normalise file form errors to match standard Drupal messages
                if (!node.querySelector('.messages__content')) {
                  const content = node.innerHTML;
                  node.innerHTML = `<div class="messages__content">${content}</div>`;
                }
                if (!node.querySelector('.messages__header')) {
                  let classMatch = [...node.classList].find(className => className.startsWith('messages--'));
                  const messageType = classMatch ? classMatch.replace('messages--', '') : 'error';
                  const header = document.createElement('div');
                  header.classList.add('messages__header');
                  header.innerHTML = '<h2 class="messages__title">' + messageType.charAt(0).toUpperCase() + messageType.slice(1) + ' message' + '</h2>';
                  node.insertBefore(header, node.firstChild);

                  //Add a polite indicator to the form item, as the message has been removed
                  const parent = mutation.target.closest('.js-form-item');
                  if(parent) {
                    parent.classList.add('st-form-managed-file--' + messageType);
                    parent.addEventListener('animationend', () => {
                      parent.classList.remove('st-form-managed-file--' + messageType);
                    });
                  }
                }
                //Run the toast messenger
                if(!this.settingsInitialised) {
                  this.getSettings();
                }
                this.messageNodes.push(node);
                node.remove();
                this.setMessageData();
                this.render();
              }
            });
          }
        });
      });
      observer.observe(element, {
        childList: true,
        subtree: false,
      });
    },
    getSettings: function () {
      const settings = drupalSettings.simple_toasts || {};
      this.messageTypeDuration = {
        status: settings.status_duration ?? this.messageTypeDuration.status,
        warning: settings.warning_duration ?? this.messageTypeDuration.warning,
        error: settings.error_duration ?? this.messageTypeDuration.error,
      };
      this.position = settings.position || this.position;
      this.animation.in = settings.animation_in || this.animation.in;
      this.animation.out = settings.animation_out || this.animation.out;
      this.style = settings.style || this.style;
      this.settingsInitialised = true;
    },
    messageObserver: function () {
      this.messageWrappers.forEach(wrapper => {
        const observer = new MutationObserver(mutationsList => {
          mutationsList.forEach(mutation => {
            if(mutation.type === 'childList' && mutation.addedNodes.length) {
              this.run();
            }
          });
        });
        observer.observe(wrapper, {
          childList: true,
          subtree: false,
        });
      });
    },
    run() {
      this.extractMessageNodes();
      if(this.messageNodes.length) {
        this.removeDomNodes();
        this.setMessageData();
        this.render();
      }
    },
    extractMessageNodes: function () {
      this.messageWrappers.forEach((messageWrapper) => {
        const childNodes = messageWrapper.childNodes;
        childNodes.forEach((node) => {
          if(node.nodeType !== 1) {
            return;
          }
          this.messageNodes.push(node);
        });
      });
    },
    setMessageData: function () {
      this.messageNodes.forEach((node, index) => {
        // Check if dataset attributes exist, otherwise derive from classList (e.g. $formState->setError is not setting them in status-messages.html.twig)
        let messageType = node.dataset.drupalMessageType;
        if(!messageType) {
          let classMatch = [...node.classList].find(className => className.startsWith('messages--'));
          messageType = classMatch ? classMatch.replace('messages--', '') : 'status';
        }
        let messageId = node.dataset.drupalMessageId || messageType + '-no-dataset-' + index; // Generate if missing

        const messageObject = {
          id: messageId,
          type: messageType,
        };

        switch (this.style) {
          case 'theme':
            //Clone the node to remove any event listeners
            messageObject.markup = node.cloneNode(true);
            const button = messageObject.markup.querySelector('button');
            if(button) {
              button.classList.add('st-toast-close');
            } else {
              //A button is must for a toast, so we must add one if there isn't already one.
              messageObject.markup.prepend(this.createCloseButton(true));
            }
            break;
          default:
            const contentElement = node.querySelector('.messages__content');
            messageObject.message = contentElement.innerHTML;
            switch (messageObject.type) {
              case 'status':
              default:
                messageObject.icon = this.iconStatus;
                messageObject.title = 'Status message';
                break;
              case 'warning':
                messageObject.icon = this.iconWarning;
                messageObject.title = 'Warning message';
                break;
              case 'error':
                messageObject.icon = this.iconError;
                messageObject.title = 'Error message';
                break;
            }
            break;
        }
        this.messages.push(messageObject);
      });
      //Now the messages array created, we can remove the original nodes
      this.messageNodes = [];
    },
    removeDomNodes: function () {
      this.messageWrappers.forEach((wrapper) => {
        wrapper.replaceChildren();
      });
    },
    render: function () {
      let toastContainer = document.querySelector('#st-toast-container');
      if(!toastContainer) {
        toastContainer = document.createElement('div');
        toastContainer.classList.add('st-toast-container', this.position, 'st-theme-' + this.style);
        toastContainer.id = 'st-toast-container';
        toastContainer.setAttribute('role', 'alert');
        toastContainer.setAttribute('aria-live', 'polite');
        toastContainer.setAttribute('aria-atomic', 'true');
        $('body').prepend(toastContainer);
      }
      const fragment = document.createDocumentFragment();
      this.messages.forEach((message, index) => {
        fragment.appendChild(this.renderItem(message, index));
      });
      toastContainer.appendChild(fragment);

      toastContainer.addEventListener('click', (e) => {
        if(e.target.closest('.st-toast-close')) {
          const toast = e.target.closest('.st-toast');
          // Remove all animate-in--X classes
          toast.className = toast.className.replace(/\banimate-in--\d+\b/g, '');

          this.hideToast(toast);
        }
      });
      //Now the rendering is complete, clear the message array
      this.messages = [];
    },
    renderItem: function (message) {
      let toast = document.createElement('div');
      toast.classList.add('st-toast', 'st-' + message.type);
      toast.id = 'st-toast-' + message.id;
      toast.setAttribute('role', 'alert');
      toast.setAttribute('aria-live', 'assertive');
      toast.setAttribute('aria-atomic', 'true');

      //Animation
      if(this.animation.in !== 'none' || this.animation.out !== 'none') {
        toast.classList.add('animate');
        if(this.animation.in !== 'none') {
          toast.classList.add('animate-in', this.animation.in);
        }
        if(this.animation.out !== 'none') {
          toast.classList.add('animate-out', this.animation.out);
        }
        //if(this.messages.length > 1) {
        toast.classList.add('animate-in--' + this.count);
        //}
      }
      toast.classList.add('st-show');
      // Handle the auto-hide functionality
      if(this.messageTypeDuration[message.type] === 0) {
        // Disable auto-hide if duration is set to 0
        toast.dataset.stAutohide = 'false';
      } else {
        toast.dataset.stAutohide = 'true';
        toast.dataset.stDelay = this.messageTypeDuration[message.type];
      }

      //Create the toast
      if(this.style === 'theme') {
        toast.appendChild(message.markup);
      } else {
        // Create the toast header
        let toastHeader = document.createElement('div');
        toastHeader.classList.add('st-toast-header', 'st-bg-' + message.type);
        toastHeader.innerHTML = '<div class="visually-hidden">' + message.title + '</div>';

        // Create the toast body
        let toastBody = document.createElement('div');
        toastBody.classList.add('st-toast-body');

        toastBody.innerHTML = '' +
          '<span class="st-toast-icon">' + message.icon + '</span>' +
          '<span class="st-toast-message">' + message.message + '</span>';
        toastBody.appendChild(this.createCloseButton());

        //Assemble toast
        toast.appendChild(toastHeader);
        toast.appendChild(toastBody);
      }
      // Auto-remove toast after delay if applicable
      if(toast.dataset.stAutohide === 'true') {
        let autoRemoveTimeout;
        // Function to start or restart the timeout
        const startTimeout = () => {
          autoRemoveTimeout = setTimeout(() => {
            Drupal.simpleToasts.hideToast(toast);
          }, parseInt(toast.dataset.stDelay, 10));
        };
        // Start the initial timeout
        startTimeout();
        // Clear the timeout on mouse enter
        toast.addEventListener('mouseenter', () => clearTimeout(autoRemoveTimeout));
        // Reapply the timeout on mouse leave
        toast.addEventListener('mouseleave', startTimeout);
      }
      this.count++;
      return toast;
    },
    hideToast(toast) {
      if(!toast.classList.contains('animate-out')) {
        toast.remove();
      } else {
        toast.classList.add('st-hide');
        toast.classList.remove('st-show');
        // Listen for the animation end event
        toast.addEventListener('animationend', () => {
          // Temporarily only animate the top aligned toasts
          if(this.position.startsWith('top')) {
            this.onToastHide(toast);
          } else {
            toast.remove();
          }
          this.count--;
        });
      }
    },
    /** Callback function for when a toast is hidden,
     * Removes the toast from the DOM and adjusts the positions of the remaining toasts
     * Handles the positions of the remaining toasts using jQuery animation
     * @TODO not used on bottom alignment, animation on multiple bottom aligned toasts needs improvement
     +*/
    onToastHide(toast) {
      // Get the container as jQuery & its dimensions
      const $container = $(toast.parentNode);
      const containerHeight = $container.height();
      const containerOuterHeight = $container.outerHeight(true);

      //Get the toast as jQuery & its dimensions
      const $toast = $(toast);
      const toastOuterHeight = $toast.outerHeight(true);
      const toastHeight = $toast.height();
      const toastMarginBottom = $toast.css('margin-bottom');

      this.totalRemovingHeight += toastOuterHeight;

      let $next = null;
      let $prev = null;
      if(this.position.startsWith('top')) {
        $next = $toast.next();
      } else if(this.position.startsWith('bottom')) {
        $prev = $toast.prev();
      }
      if(($next && $next.length) || ($prev && $prev.length)) {
        $container.height(containerHeight);
      }
      toast.remove();
      //Top positioned toasts
      if($next && $next.length) {
        $next.css({ 'margin-top': toastHeight });
        $next.animate({ 'margin-top': 0 }, this.animation.duration, 'swing', function () {
          $container.css('height', '');
          //Not strictly required here but resetting the totalRemovingHeight
          Drupal.simpleToasts.totalRemovingHeight = 0;
        });
      }
      //Bottom positioned toasts
      else if($prev && $prev.length) {
        const removingCount = $('.toast-container .toast.removing').length;
        $prev.css({ 'margin-bottom': toastHeight });
        $prev.animate({ 'margin-bottom': toastMarginBottom }, this.animation.duration, 'swing', function () {
          if(removingCount === 0) {
            $container.animate({ height: (containerOuterHeight - Drupal.simpleToasts.totalRemovingHeight) }, Drupal.simpleToasts.animation.duration, 'swing', function () {
              $container.css('height', '');
              Drupal.simpleToasts.totalRemovingHeight = 0;
            });
          }
        });
      }
    },
    createCloseButton(themeGenerated = false) {
      const button = document.createElement('button');
      button.innerHTML = this.iconClose;
      button.type = 'button';
      button.classList.add('st-toast-close');
      if(themeGenerated) {
        button.classList.add('st-theme-generated');
      }
      button.ariaLabel = 'Close';
      return button;
    },
  };

  /**
   * Registers behaviours
   */
  Drupal.behaviors.simpleToasts = {
    attach: function (context, settings) {
      //Capture standard Drupal messages
      once(Drupal.simpleToasts.id, Drupal.simpleToasts.context, context).forEach(function (element) {
        Drupal.simpleToasts.init(element);
      });
      //Capture javascript fallback messages
      once(Drupal.simpleToasts.id, Drupal.simpleToasts.fallbackContext, context).forEach(function (element) {
        Drupal.simpleToasts.initJsFallback(element);
      });
      //Capture managed file form messages
      once(Drupal.simpleToasts.id, Drupal.simpleToasts.managedFileContext, context).forEach(function (element) {
        Drupal.simpleToasts.initManagedFile(element);
      });
    },
  };

}(jQuery, Drupal, once, drupalSettings));
