<?php

namespace Drupal\simple_toasts\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Simple Toasts general settings.
 */
class SimpleToastsSettingsForm extends ConfigFormBase {

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected ThemeHandlerInterface $themeHandler;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *  The typed configuration manager.
   * @param \Drupal\Core\Extension\ThemeHandlerInterface $theme_handler
   *   The theme handler.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, ThemeHandlerInterface $theme_handler) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->themeHandler = $theme_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('theme_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'simple_toasts_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'simple_toasts.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('simple_toasts.settings');
    $current_themes = $config->get('themes') ?: [];

    $form['themes_container'] = [
      '#type' => 'container',
      '#title' => $this->t('Theme Configurations'),
      '#tree' => TRUE,
    ];

    // Get all available themes
    $theme_options = $this->getThemeOptions();

    // Iterate through themes and create form elements
    foreach ($theme_options as $theme_machine_name => $theme_name) {
      // Determine current style, default to 'default' if not set
      $current_style = $current_themes[$theme_machine_name]['style'] ?? 'default';
      // Determine current position, default to 'top-right' if not set
      $current_position = $current_themes[$theme_machine_name]['position'] ?? 'top-right';

      $form['themes_container'][$theme_machine_name] = [
        '#type' => 'details',
        '#title' => $this->t('@theme_name Theme', ['@theme_name' => $theme_name]),
        '#open' => FALSE,
      ];

      $form['themes_container'][$theme_machine_name]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable toast messages for @theme_name', ['@theme_name' => $theme_name]),
        '#default_value' => isset($current_themes[$theme_machine_name]) ? 1 : 0,
      ];

      $form['themes_container'][$theme_machine_name]['style'] = [
        '#type' => 'select',
        '#title' => $this->t('Toast Message Style'),
        '#description' => $this->t('Choose how toast messages should be styled for the @theme_name theme. \'Light\', \'Dark\' & \'Vivid\' are styles provided by Simple Toasts, or use the theme\'s messaging style', ['@theme_name' => $theme_name]),
        '#options' => [
          'default' => $this->t('Light'),
          'dark' => $this->t('Dark'),
          'vivid' => $this->t('Vivid'),
          'theme' => $this->t('@theme_name', ['@theme_name' => $theme_name]),
        ],
        '#default_value' => $current_style,
        '#states' => [
          'visible' => [
            ':input[name="themes_container[' . $theme_machine_name . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['themes_container'][$theme_machine_name]['position'] = [
        '#type' => 'select',
        '#title' => $this->t('Message Position'),
        '#description' => $this->t('The position of the messages on the screen for the @theme_name theme.', ['@theme_name' => $theme_name]),
        '#options' => [
          'top-right' => $this->t('Top Right'),
          'top-center' => $this->t('Top Center'),
          'top-left' => $this->t('Top Left'),
          'bottom-right' => $this->t('Bottom Right'),
          'bottom-center' => $this->t('Bottom Center'),
          'bottom-left' => $this->t('Bottom Left'),
        ],
        '#default_value' => $current_position,
        '#states' => [
          'visible' => [
            ':input[name="themes_container[' . $theme_machine_name . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

    }

    $form['duration'] = [
      '#type' => 'details',
      '#title' => $this->t('Duration'),
      '#open' => TRUE,
    ];

    $form['duration']['status_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Status message duration'),
      '#default_value' => $config->get('status_duration'),
      '#size' => 40,
      '#description' => $this->t('The duration in milliseconds before status messages disappear. Set to 0 to disable auto-dismiss.'),
    ];
    $form['duration']['warning_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Warning message duration'),
      '#default_value' => $config->get('warning_duration'),
      '#size' => 40,
      '#description' => $this->t('The duration in milliseconds before warning messages disappear. Set to 0 to disable auto-dismiss.'),
    ];
    $form['duration']['error_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Error message duration'),
      '#default_value' => $config->get('error_duration'),
      '#size' => 40,
      '#description' => $this->t('The duration in milliseconds before error messages disappear. Set to 0 to disable auto-dismiss.'),
    ];

    $form['animation'] = [
      '#type' => 'details',
      '#title' => $this->t('Animation'),
      '#open' => TRUE,
    ];

    $form['animation']['animation_in'] = [
      '#type' => 'select',
      '#title' => $this->t('Animation in'),
      '#default_value' => $config->get('animation_in'),
      '#description' => $this->t('Animate the toast in.'),
      '#options' => [
        'slide-in' => $this->t('Slide'),
        'fade-in' => $this->t('Fade'),
        'none' => $this->t('None'),
      ],
    ];

    $form['animation']['animation_out'] = [
      '#type' => 'select',
      '#title' => $this->t('Animation out'),
      '#default_value' => $config->get('animation_out'),
      '#description' => $this->t('Animate the toast out.'),
      '#options' => [
        'slide-out' => $this->t('Slide'),
        'fade-out' => $this->t('Fade'),
        'none' => $this->t('None'),
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Process the themes configuration
    $themes_config = [];
    $themes_container = $form_state->getValue('themes_container');

    foreach ($themes_container as $theme_machine_name => $theme_data) {
      // Only add themes that are enabled
      if ($theme_data['enabled']) {
        $themes_config[$theme_machine_name] = [
          'name' => $this->getThemeOptions()[$theme_machine_name],
          'style' => $theme_data['style'],
          'position' => $theme_data['position'],
        ];
      }
    }

    $this->configFactory->getEditable('simple_toasts.settings')
      ->set('themes', $themes_config)
      ->set('status_duration', $form_state->getValue('status_duration'))
      ->set('warning_duration', $form_state->getValue('warning_duration'))
      ->set('error_duration', $form_state->getValue('error_duration'))
      ->set('animation_in', $form_state->getValue('animation_in'))
      ->set('animation_out', $form_state->getValue('animation_out'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Return an array with all the themes.
   *
   * @return array
   *   An array with all the themes.
   */
  protected function getThemeOptions(): array {
    $output = [];
    foreach ($this->themeHandler->listInfo() as $key => $value) {
      //$output[$key] = $value->getName();
      $output[$key] = $value->info['name'];
    }
    return $output;
  }
}
