<?php

namespace Drupal\simplesamlphp_sp\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Config form for the SimpleSAMLphp SP module.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Special option key used to disable role restrictions.
   *
   * @var string
   */
  private const NONE_OPTION_KEY = '_none';

  /**
   * User role storage service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * SettingsForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Provides access to configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   Typed configuration manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Used to load available user roles.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simplesamlphp_sp_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['simplesamlphp_sp.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('simplesamlphp_sp.settings');

    $role_options = [];
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    foreach ($roles as $role) {
      if ($role->id() === RoleInterface::ANONYMOUS_ID) {
        continue;
      }
      $role_options[$role->id()] = $role->label();
    }
    asort($role_options);

    $none_option_key = static::NONE_OPTION_KEY;
    $role_options = [$none_option_key => $this->t('None (allow all roles)')] + $role_options;

    $form['activate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Activate authentication via SimpleSAMLphp'),
      '#description' => $this->t('If checked, the SSO authentication will be enabled.'),
      '#default_value' => (bool) $config->get('activate'),
    ];

    $form['simplesamlphp_base_dir'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SimpleSAMLphp base path'),
      '#description' => $this->t('Full filesystem path to SimpleSAMLphp installation (directory containing config/, modules/, www/).'),
      '#default_value' => $config->get('simplesamlphp_base_dir') ?: '/var/simplesamlphp',
      '#required' => FALSE,
    ];

    $form['sp_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Service Provider name'),
      '#description' => $this->t('SP name as configured in SimpleSAMLphp (e.g. "default-sp").'),
      '#default_value' => $config->get('sp_name') ?: 'default-sp',
      '#required' => TRUE,
    ];

    $form['saml_login_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SAML login path'),
      '#description' => $this->t('Public path on this Drupal site that triggers SAML login (e.g. /saml/login). This can be any valid path.'),
      '#default_value' => $config->get('saml_login_path') ?: '/saml/login',
      '#required' => TRUE,
    ];

    $form['user_info_syncing'] = [
      '#type' => 'details',
      '#title' => $this->t('User info and syncing'),
      '#open' => TRUE,
    ];

    $form['user_info_syncing']['unique_id_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unique identifier attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as unique identifier for the user (e.g. eduPersonPrincipalName, uid).'),
      '#default_value' => $config->get('unique_id_attribute') ?: 'uid',
      '#required' => TRUE,
    ];

    $form['user_info_syncing']['username_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as username for the user (e.g. uid).'),
      '#default_value' => $config->get('username_attribute') ?: 'uid',
      '#required' => TRUE,
    ];

    $form['user_info_syncing']['email_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as email address for the user (e.g. mail).'),
      '#default_value' => $config->get('email_attribute') ?: 'mail',
      '#required' => TRUE,
    ];

    $default_blocked_roles = $config->get('blocked_roles');
    if (!is_array($default_blocked_roles)) {
      $default_blocked_roles = [];
    }
    if (empty($default_blocked_roles)) {
      $default_blocked_roles = [$none_option_key];
    }

    $form['drupal_native_account'] = [
      '#type' => 'details',
      '#title' => $this->t('Drupal native account'),
      '#open' => TRUE,
    ];

    $form['drupal_native_account']['blocked_roles'] = [
      '#type' => 'select',
      '#title' => $this->t('Roles denied SAML login'),
      '#description' => $this->t('Users belonging to any selected role cannot authenticate via SAML.'),
      '#options' => $role_options,
      '#default_value' => $default_blocked_roles,
      '#multiple' => TRUE,
    ];

    $form['drupal_native_account']['lock_external_user_fields'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prevent editing Drupal credentials for externally created accounts'),
      '#description' => $this->t('When enabled, accounts provisioned by SimpleSAMLphp cannot change their username, email, or password through Drupal.'),
      '#default_value' => (bool) $config->get('lock_external_user_fields'),
    ];

    $allow_native_default = (bool) $config->get('allow_native_login_for_external_users');
    if ((bool) $config->get('lock_external_user_fields')) {
      $allow_native_default = FALSE;
    }

    $form['drupal_native_account']['allow_native_login_for_external_users'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Drupal native login for SAML-provisioned accounts'),
      '#description' => $this->t('Controls whether accounts created by SimpleSAMLphp may authenticate using Drupal passwords or one-time login links.'),
      '#default_value' => $allow_native_default,
      '#states' => [
        'visible' => [
          ':input[name="lock_external_user_fields"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $exempt_ids = $config->get('exempt_user_ids');
    if (!is_array($exempt_ids)) {
      $exempt_ids = [];
    }
    $exempt_ids[] = 1;
    $exempt_ids = array_values(array_unique(array_map('intval', array_filter($exempt_ids))));
    $exempt_users = $this->entityTypeManager->getStorage('user')->loadMultiple($exempt_ids);

    $form['drupal_native_account']['exempt_user_ids'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Exempt users'),
      '#description' => $this->t('Selected users(Comma-separated list) can edit credentials and use Drupal native login even when restrictions are enabled. User 1 is always exempt.'),
      '#target_type' => 'user',
      '#tags' => TRUE,
      '#default_value' => $exempt_users,
    ];

    $form['#attached']['library'][] = 'simplesamlphp_sp/form_helpers';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $lock_external = (bool) $form_state->getValue('lock_external_user_fields');
    $allow_native_login = !$lock_external && (bool) $form_state->getValue('allow_native_login_for_external_users');
    $form_state->setValue('allow_native_login_for_external_users', $allow_native_login);
    $exempt_user_ids = $this->sanitizeExemptUserIds((array) $form_state->getValue('exempt_user_ids'));

    $this->config('simplesamlphp_sp.settings')
      ->set('activate', $form_state->getValue('activate'))
      ->set('simplesamlphp_base_dir', $form_state->getValue('simplesamlphp_base_dir'))
      ->set('sp_name', $form_state->getValue('sp_name'))
      ->set('saml_login_path', $form_state->getValue('saml_login_path'))
      ->set('unique_id_attribute', $form_state->getValue('unique_id_attribute'))
      ->set('username_attribute', $form_state->getValue('username_attribute'))
      ->set('email_attribute', $form_state->getValue('email_attribute'))
      ->set('blocked_roles', $this->sanitizeBlockedRoles((array) $form_state->getValue('blocked_roles')))
      ->set('lock_external_user_fields', $lock_external)
      ->set('allow_native_login_for_external_users', $allow_native_login)
      ->set('exempt_user_ids', $exempt_user_ids)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Normalizes the blocked_roles value before persisting it.
   *
   * @param array $selected_roles
   *   Selected role identifiers from the form submission.
   *
   * @return array
   *   Sanitized list of role IDs to store in configuration.
   */
  protected function sanitizeBlockedRoles(array $selected_roles): array {
    $none_option_key = static::NONE_OPTION_KEY;
    $selected_roles = array_values(array_filter($selected_roles));

    if (in_array($none_option_key, $selected_roles, TRUE)) {
      return [];
    }

    return $selected_roles;
  }

  /**
   * Normalizes the exempt user list before saving configuration.
   */
  protected function sanitizeExemptUserIds(array $selected_users): array {
    $ids = [];
    foreach ($selected_users as $value) {
      if (is_array($value) && isset($value['target_id'])) {
        $ids[] = (int) $value['target_id'];
      }
      elseif (is_numeric($value)) {
        $ids[] = (int) $value;
      }
    }

    $ids[] = 1;
    $ids = array_unique(array_filter($ids, static fn($value) => $value > 0));
    sort($ids, SORT_NUMERIC);

    return array_values($ids);
  }

}
