<?php

namespace Drupal\simplesamlphp_sp\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper;
use Drupal\simplesamlphp_sp\Service\SimpleSamlManager;
use Drupal\user\UserInterface;

/**
 * Hook implementations for the SimpleSAMLphp SP module.
 */
class SimplesamlphpSpHooks {
  use StringTranslationTrait;

  /**
   * The SimpleSAMLphp SP manager.
   */
  protected SimpleSamlManager $samlManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger channel for this module.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Entity type manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Helper for identifying SAML-managed accounts.
   */
  protected SimpleSamlAccountHelper $accountHelper;

  public function __construct(SimpleSamlManager $samlManager, ConfigFactoryInterface $configFactory, LoggerChannelInterface $logger, EntityTypeManagerInterface $entityTypeManager, SimpleSamlAccountHelper $accountHelper) {
    $this->samlManager = $samlManager;
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->entityTypeManager = $entityTypeManager;
    $this->accountHelper = $accountHelper;
  }

  /**
   * Implements hook_user_logout().
   */
  #[Hook('user_logout')]
  public function userLogout(AccountInterface $account): void {
    $user = \Drupal::currentUser();
    // Keep the user id before forcing a logout.
    $user_id = $user->id();

    // Drupal normally invalidates the session before this hook fires, but we
    // repeat the step to avoid any ambiguity if the IdP redirect cuts the
    // logout flow short.
    if ($user->isAuthenticated()) {
      $this->logger->info('Invalidating session before initiating SAML logout.');
      $request = \Drupal::request();
      $session = $request->getSession();
      $session->invalidate();
      $user->setAccount(new AnonymousUserSession());
    }

    // When SSO feature is inactive, do nothing.
    if (!$this->samlManager->isActivated()) {
      $this->logger->info('SimpleSAMLphp SSO is disabled; skipping SAML logout.');
      return;
    }
    // Start the SSO logout process to IDP.
    $this->logger->info('Initiating SAML logout for user @uid.', ['@uid' => $user_id]);
    $this->samlManager->logout();
  }

  /**
   * Implements hook_user_presave().
   */
  #[Hook('user_presave')]
  public function userPresave(UserInterface $account): void {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    if (!$config->get('lock_external_user_fields')) {
      return;
    }

    if ($account->isNew()) {
      return;
    }

    if (!$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $original_account = $account->original ?? $this->entityTypeManager->getStorage('user')->load($account->id());
    if (!$original_account instanceof UserInterface) {
      return;
    }

    $blocked_fields = [];

    if ($account->getAccountName() !== $original_account->getAccountName()) {
      $account->setUsername($original_account->getAccountName());
      $blocked_fields[] = 'username';
    }

    if ($account->getEmail() !== $original_account->getEmail()) {
      $account->setEmail($original_account->getEmail());
      $blocked_fields[] = 'email';
    }

    $current_password = $account->getPassword();
    $original_password = $original_account->getPassword();
    if ($current_password !== $original_password) {
      $account->get('pass')->value = $original_password;
      $blocked_fields[] = 'password';
    }

    if ($blocked_fields) {
      $this->logger->info('Prevented updates to @fields for externally managed user account @uid.', [
        '@fields' => implode(', ', $blocked_fields),
        '@uid' => $account->id(),
      ]);
    }
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    $lockCredentials = (bool) $config->get('lock_external_user_fields');
    $restrictNativeLogin = $this->shouldRestrictNativeLogin($config);

    $form_object = $form_state->getFormObject();
    if ($form_object instanceof EntityFormInterface) {
      $entity = $form_object->getEntity();
      if ($lockCredentials && $entity instanceof UserInterface && !$entity->isNew() && $this->accountHelper->isSamlAccount($entity) && !$this->accountHelper->isExempt($entity)) {
        $this->restrictCredentialFields($form);
      }
      return;
    }

    switch ($form_id) {
      case 'user_login_form':
      case 'user_login_block':
        if ($restrictNativeLogin) {
          $form['#validate'][] = [$this, 'validateNativeLoginRestriction'];
        }
        break;

      case 'user_pass':
        if ($restrictNativeLogin) {
          $form['#validate'][] = [$this, 'validateNativePasswordRequest'];
        }
        break;

      case 'user_pass_reset':
        if ($restrictNativeLogin) {
          $this->alterPasswordResetForm($form, $form_state);
        }
        break;
    }
  }

  /**
   * Hides credential-related elements on the user account edit form.
   */
  protected function restrictCredentialFields(array &$form): void {
    $targets = [
      ['account', 'name'],
      ['account', 'mail'],
      ['account', 'pass'],
      ['account', 'current_pass'],
    ];

    foreach ($targets as $path) {
      $this->setElementAccess($form, $path, FALSE);
    }
  }

  /**
   * Determines whether Drupal-native login must be blocked for SAML accounts.
   */
  protected function shouldRestrictNativeLogin(?ImmutableConfig $config = NULL): bool {
    $config = $config ?: $this->configFactory->get('simplesamlphp_sp.settings');
    return (bool) $config->get('lock_external_user_fields') && !$config->get('allow_native_login_for_external_users');
  }

  /**
   * Validation handler preventing native login for SAML-managed accounts.
   */
  public function validateNativeLoginRestriction(array &$form, FormStateInterface $form_state): void {
    if (!$this->shouldRestrictNativeLogin()) {
      return;
    }

    $uid = $form_state->get('uid');
    if (empty($uid)) {
      return;
    }

    $account = $this->accountHelper->loadAccountById((int) $uid);
    if (!$account instanceof UserInterface || !$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $form_state->set('uid', FALSE);
    $form_state->setErrorByName('name', $this->t('This account must sign in using Single Sign-On.'));
  }

  /**
   * Validation handler preventing password resets for SAML-managed accounts.
   */
  public function validateNativePasswordRequest(array &$form, FormStateInterface $form_state): void {
    if (!$this->shouldRestrictNativeLogin()) {
      return;
    }

    $account = $form_state->getValue('account');
    if (!$account instanceof UserInterface) {
      return;
    }

    if (!$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $form_state->setValue('account', NULL);
    $form_state->setErrorByName('name', $this->t('Password resets are disabled for accounts managed by Single Sign-On.'));
  }

  /**
   * Alters password reset confirmation forms for SAML-managed accounts.
   */
  protected function alterPasswordResetForm(array &$form, FormStateInterface $form_state): void {
    $account = $this->extractPasswordResetAccount($form_state);
    if (!$account instanceof UserInterface) {
      return;
    }

    if (!$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $form['simplesamlphp_sp_native_login_block'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--error']],
      '#weight' => -10,
      'message' => ['#markup' => $this->t('Password reset links cannot be used with accounts managed by Single Sign-On.')],
    ];

    $this->setElementAccess($form, ['actions', 'submit'], FALSE);
  }

  /**
   * Extracts the account associated with a password reset form build.
   */
  protected function extractPasswordResetAccount(FormStateInterface $form_state): ?UserInterface {
    $build_info = $form_state->getBuildInfo();
    $args = $build_info['args'] ?? [];
    $account = $args[0] ?? NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Sets the #access property on a nested form element when it exists.
   */
  protected function setElementAccess(array &$form, array $path, bool $value): void {
    $element = &$form;
    foreach ($path as $segment) {
      if (!isset($element[$segment])) {
        unset($element);
        return;
      }
      $element = &$element[$segment];
    }
    $element['#access'] = $value;
    unset($element);
  }

}
