<?php

namespace Drupal\simplesamlphp_sp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\externalauth\AuthmapInterface;
use Drupal\user\UserInterface;

/**
 * Helper for identifying accounts managed by SimpleSAMLphp.
 */
class SimpleSamlAccountHelper {

  public function __construct(
    protected AuthmapInterface $authmap,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Loads a user account by identifier.
   */
  public function loadAccountById(int $uid): ?UserInterface {
    $account = $this->entityTypeManager->getStorage('user')->load($uid);
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Loads a user account by username.
   */
  public function loadAccountByName(string $name): ?UserInterface {
    $accounts = $this->entityTypeManager->getStorage('user')->loadByProperties(['name' => $name]);
    $account = reset($accounts) ?: NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Loads a user account by email address.
   */
  public function loadAccountByMail(string $mail): ?UserInterface {
    $accounts = $this->entityTypeManager->getStorage('user')->loadByProperties(['mail' => $mail]);
    $account = reset($accounts) ?: NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Determines whether the provided account is maintained by SimpleSAMLphp.
   */
  public function isSamlAccount(UserInterface $account): bool {
    if ($account->isNew()) {
      return FALSE;
    }

    $authmap_data = $this->authmap->getAuthData((int) $account->id(), SimpleSamlSpAuth::PROVIDER_ID);
    if ($authmap_data === FALSE) {
      $authmap_data = NULL;
    }
    return $this->isSamlAccountFromAuthmap($account, $authmap_data);
  }

  /**
   * Determines whether the provided account is exempt from restrictions.
   */
  public function isExempt(UserInterface $account): bool {
    return in_array((int) $account->id(), $this->getExemptUserIds(), TRUE);
  }

  /**
   * Determines whether the provided account identifier is exempt.
   */
  public function isExemptById(int $uid): bool {
    return in_array($uid, $this->getExemptUserIds(), TRUE);
  }

  /**
   * Determines whether the provided account identifier is managed by SAML.
   */
  public function isSamlAccountById(int $uid): bool {
    $account = $this->loadAccountById($uid);
    return $account ? $this->isSamlAccount($account) : FALSE;
  }

  /**
   * Retrieves the configured exempt user IDs.
   */
  public function getExemptUserIds(): array {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    $ids = $config->get('exempt_user_ids');
    if (!is_array($ids)) {
      $ids = [];
    }
    $ids[] = 1;
    $ids = array_unique(array_filter(array_map('intval', $ids), static fn($value) => $value > 0));
    sort($ids, SORT_NUMERIC);
    return array_values($ids);
  }

  /**
   * Determines whether the provided account is provisioned by SAML.
   */
  protected function isSamlAccountFromAuthmap(UserInterface $account, ?array $authmap_data): bool {
    if (!empty($authmap_data['data'])) {
      $flags = $this->decodeAuthmapFlags($authmap_data['data']);
      if (!empty($flags[SimpleSamlSpAuth::AUTHMAP_CREATED_FLAG])) {
        return TRUE;
      }
    }

    $init = (string) $account->get('init')->value;
    return str_starts_with($init, SimpleSamlSpAuth::PROVIDER_ID . '_');
  }

  /**
   * Safely decodes serialized authmap flags.
   */
  protected function decodeAuthmapFlags(mixed $data): array {
    if (!is_string($data) || $data === '') {
      return [];
    }

    try {
      $flags = unserialize($data, ['allowed_classes' => FALSE]);
      return is_array($flags) ? $flags : [];
    }
    catch (\Throwable $throwable) {
      return [];
    }
  }

}
