<?php

namespace Drupal\simplesamlphp_sp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use SimpleSAML\Error\CriticalConfigurationError;
use SimpleSAML\Session;

/**
 * Coordinates interactions with the SimpleSAMLphp service provider instance.
 */
class SimpleSamlManager {
  use StringTranslationTrait;

  /**
   * The module configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The current user account.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * Cached SimpleSAMLphp client instance.
   *
   * @var \Drupal\simplesamlphp_sp\Service\SimpleSamlAuthClientInterface|null
   */
  protected $spInstance;

  /**
   * Messenger service for surfacing configuration errors.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Cached attribute set pulled from the IdP.
   *
   * @var array|null
   */
  protected $attributes;

  /**
   * Factory for creating SimpleSAML client wrappers.
   */
  protected SimpleSamlAuthClientFactoryInterface $clientFactory;

  /**
   * Constructs the manager and boots the SimpleSAMLphp library.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Provides access to module configuration.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger service for reporting configuration issues.
   * @param \Drupal\simplesamlphp_sp\Service\SimpleSamlAuthClientFactoryInterface $client_factory
   *   Factory responsible for producing SimpleSAML clients.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountInterface $account, MessengerInterface $messenger, SimpleSamlAuthClientFactoryInterface $client_factory) {
    $this->config = $config_factory->get('simplesamlphp_sp.settings');
    $this->account = $account;
    $this->messenger = $messenger;
    $this->clientFactory = $client_factory;
  }

  /**
   * Checks if SimpleSAMLphp_auth is enabled.
   *
   * @return bool
   *   Whether SimpleSAMLphp authentication is enabled or not.
   */
  public function isActivated() {
    return (bool) $this->config->get('activate');
  }

  /**
   * Gets all SimpleSAML attributes.
   *
   * @return array
   *   Array of SimpleSAML attributes.
   */
  public function getAttributes() {
    if (!$this->attributes) {
      $client = $this->getSimpleInstance();
      if ($client === NULL) {
        return [];
      }
      $this->attributes = $client->getAttributes();
    }
    return $this->attributes;
  }

  /**
   * Get a specific SimpleSAML attribute.
   *
   * @param string $attribute
   *   The name of the attribute.
   *
   * @return mixed|bool
   *   The attribute value or FALSE.
   *
   * @throws \Drupal\simplesamlphp_auth\Exception\SimplesamlphpAttributeException
   *   Exception when attribute is not set.
   */
  public function getAttribute($attribute) {
    $attributes = $this->getAttributes();

    if (isset($attributes)) {
      if (!empty($attributes[$attribute][0])) {
        return $attributes[$attribute][0];
      }
    }

    return NULL;
  }

  /**
   * Gets the unique id of the user from the IdP.
   *
   * @return string
   *   The authname.
   */
  public function getAuthname() {
    $unique_id = $this->config->get('unique_id');
    if (empty($unique_id)) {
      $unique_id = 'email';
    }
    return $this->getAttribute($unique_id);
  }

  /**
   * Forces the current request through the IdP authentication flow.
   *
   * @return array
   *   Attribute set returned by the IdP.
   */
  public function requireAuth(): array {
    $auth = $this->getSimpleInstance();
    if ($auth === NULL) {
      return [];
    }
    // This requireAuth() function will only return
    // if the user is authenticated.
    // If the user isn't authenticated,
    // this function will start the authentication process.
    $result = $auth->requireAuth();
    // Save the current session and clean any left overs
    // that could interfere with the normal application behaviour.
    Session::getSessionFromRequest()->cleanup();

    return $result;
  }

  /**
   * Returns or builds the configured SimpleSAMLphp client.
   *
   * @return \Drupal\simplesamlphp_sp\Service\SimpleSamlAuthClientInterface|null
   *   The initialized client or NULL if configuration errors block it.
   */
  protected function getSimpleInstance(): ?SimpleSamlAuthClientInterface {
    if (!empty($this->spInstance)) {
      return $this->spInstance;
    }

    $sp = $this->config->get('sp_name') ?: 'default-sp';
    try {
      $this->spInstance = $this->clientFactory->create($sp);
      return $this->spInstance;
    }
    catch (CriticalConfigurationError $e) {
      if ($this->account->hasPermission('administer simplesamlphp_sp configuration')) {
        $this->messenger->addError($this->t('There is a Simplesamlphp configuration problem. @message', ['@message' => $e->getMessage()]), 'error');
      }
      return NULL;
    }
  }

  /**
   * Logs the user out of the IdP and clears the local SimpleSAML session.
   *
   * @param string|null $return_to
   *   Absolute URL the IdP should redirect to after logout. Defaults to front.
   */
  public function logout($return_to = NULL) {
    $auth = $this->getSimpleInstance();
    if ($auth === NULL) {
      return;
    }

    if ($return_to === NULL) {
      $return_to = base_path();
    }
    $auth->logout($return_to);
    Session::getSessionFromRequest()->cleanup();
  }

}
