<?php

namespace Drupal\simplesamlphp_sp\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\simplesamlphp_sp\Service\SimpleSamlAuthClientFactoryInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use SimpleSAML\Auth\Simple;
use SimpleSAML\Module\saml\Auth\Source\SP;

/**
 * Config form for the SimpleSAMLphp SP module.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Special option key used to disable role restrictions.
   *
   * @var string
   */
  private const NONE_OPTION_KEY = '_none';

  /**
   * User role storage service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Factory used to build SimpleSAML clients for configuration testing.
   */
  protected SimpleSamlAuthClientFactoryInterface $simpleSamlAuthClientFactory;

  /**
   * SettingsForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Provides access to configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   Typed configuration manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Used to load available user roles.
   * @param \Drupal\simplesamlphp_sp\Service\SimpleSamlAuthClientFactoryInterface $simple_saml_auth_client_factory
   *   Factory responsible for creating SimpleSAML clients during testing.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityTypeManagerInterface $entity_type_manager, SimpleSamlAuthClientFactoryInterface $simple_saml_auth_client_factory) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
    $this->simpleSamlAuthClientFactory = $simple_saml_auth_client_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('simplesamlphp_sp.simple_saml_client_factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'simplesamlphp_sp_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['simplesamlphp_sp.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('simplesamlphp_sp.settings');

    $preview_url = $form_state->get('simplesamlphp_sp_test_preview_url');
    $preview_description = $form_state->get('simplesamlphp_sp_test_preview_description');
    if (is_string($preview_url) && $preview_url !== '') {
      $form['test_idp_preview'] = [
        '#type' => 'container',
        '#weight' => -100,
        '#attributes' => ['class' => ['simplesamlphp-sp-test-preview']],
        'description' => [
          '#markup' => is_string($preview_description) && $preview_description !== ''
            ? $preview_description
            : $this->t('The Identity Provider metadata is displayed below for verification.'),
        ],
        'iframe' => [
          '#type' => 'html_tag',
          '#tag' => 'iframe',
          '#value' => '',
          '#attributes' => [
            'src' => $preview_url,
            'width' => '100%',
            'height' => '600',
            'style' => 'border:1px solid #d2d2d2;',
            'loading' => 'eager',
          ],
        ],
      ];
    }

    $role_options = [];
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    foreach ($roles as $role) {
      if ($role->id() === RoleInterface::ANONYMOUS_ID) {
        continue;
      }
      $role_options[$role->id()] = $role->label();
    }
    asort($role_options);

    $none_option_key = static::NONE_OPTION_KEY;
    $role_options = [$none_option_key => $this->t('None (allow all roles)')] + $role_options;

    $form['activate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Activate authentication via SimpleSAMLphp'),
      '#description' => $this->t('If checked, the SSO authentication will be enabled.'),
      '#default_value' => (bool) $config->get('activate'),
    ];

    $form['simplesamlphp_base_dir'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SimpleSAMLphp base path'),
      '#description' => $this->t('Absolute path to the SimpleSAMLphp installation directory (the directory containing config/, lib/, and public/).
      <br>This setting is ignored when the <b>SIMPLESAMLPHP_INSTALL_DIR</b> environment variable is defined.
      <br>Leave this field empty when SimpleSAMLphp is installed via Composer.'),
      '#default_value' => $config->get('simplesamlphp_base_dir') ?? '/var/simplesamlphp',
      '#required' => FALSE,
    ];

    $form['sp_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Service Provider name'),
      '#description' => $this->t('SP name as configured in SimpleSAMLphp (e.g. "default-sp").'),
      '#default_value' => $config->get('sp_name') ?: 'default-sp',
      '#required' => TRUE,
    ];

    $form['saml_login_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SAML login path'),
      '#description' => $this->t('Public path on this Drupal site that triggers SAML login (e.g. /saml/login). This can be any valid path.'),
      '#default_value' => $config->get('saml_login_path') ?: '/saml/login',
      '#required' => TRUE,
    ];

    $form['user_info_syncing'] = [
      '#type' => 'details',
      '#title' => $this->t('User info and syncing'),
      '#open' => TRUE,
    ];

    $form['user_info_syncing']['unique_id_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Unique identifier attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as unique identifier for the user (e.g. eduPersonPrincipalName, uid).'),
      '#default_value' => $config->get('unique_id_attribute') ?: 'uid',
      '#required' => TRUE,
    ];

    $form['user_info_syncing']['username_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as username for the user (e.g. uid).'),
      '#default_value' => $config->get('username_attribute') ?: 'uid',
      '#required' => TRUE,
    ];

    $form['user_info_syncing']['email_attribute'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email attribute'),
      '#description' => $this->t('SimpleSAMLphp attribute to be used as email address for the user (e.g. mail).'),
      '#default_value' => $config->get('email_attribute') ?: 'mail',
      '#required' => TRUE,
    ];

    $default_blocked_roles = $config->get('blocked_roles');
    if (!is_array($default_blocked_roles)) {
      $default_blocked_roles = [];
    }
    if (empty($default_blocked_roles)) {
      $default_blocked_roles = [$none_option_key];
    }

    $form['drupal_native_account'] = [
      '#type' => 'details',
      '#title' => $this->t('Drupal native account'),
      '#open' => TRUE,
    ];

    $form['drupal_native_account']['blocked_roles'] = [
      '#type' => 'select',
      '#title' => $this->t('Roles denied SAML login'),
      '#description' => $this->t('Users belonging to any selected role cannot authenticate via SAML.'),
      '#options' => $role_options,
      '#default_value' => $default_blocked_roles,
      '#multiple' => TRUE,
    ];

    $form['drupal_native_account']['lock_external_user_fields'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prevent editing Drupal credentials for externally created accounts'),
      '#description' => $this->t('When enabled, accounts provisioned by SimpleSAMLphp cannot change their username, email, or password through Drupal.'),
      '#default_value' => (bool) $config->get('lock_external_user_fields'),
    ];

    $allow_native_default = (bool) $config->get('allow_native_login_for_external_users');

    $form['drupal_native_account']['allow_native_login_for_external_users'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow Drupal native login for SAML-provisioned accounts'),
      '#description' => $this->t('Controls whether accounts created by SimpleSAMLphp may authenticate using Drupal passwords or one-time login links.'),
      '#default_value' => $allow_native_default,
      '#states' => [
        'visible' => [
          ':input[name="lock_external_user_fields"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $exempt_ids = $config->get('exempt_user_ids');
    if (!is_array($exempt_ids)) {
      $exempt_ids = [];
    }
    $exempt_ids[] = 1;
    $exempt_ids = array_values(array_unique(array_map('intval', array_filter($exempt_ids))));
    $exempt_users = $this->entityTypeManager->getStorage('user')->loadMultiple($exempt_ids);

    $form['drupal_native_account']['exempt_user_ids'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Exempt users'),
      '#description' => $this->t('Selected users(Comma-separated list) can edit credentials and use Drupal native login even when restrictions are enabled. User 1 is always exempt.'),
      '#target_type' => 'user',
      '#tags' => TRUE,
      '#default_value' => $exempt_users,
    ];

    $form['logging'] = [
      '#type' => 'details',
      '#title' => $this->t('Logging'),
      '#open' => FALSE,
    ];

    $form['logging']['log_debug_information'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Record debug information'),
      '#description' => $this->t('If checked, debug-level messages generated by the module are recorded in the log.'),
      '#default_value' => (bool) $config->get('log_debug_information'),
    ];

    $form['#attached']['library'][] = 'simplesamlphp_sp/form_helpers';

    $form = parent::buildForm($form, $form_state);

    $form['actions']['test_idp'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test IDP'),
      '#submit' => ['::submitTestIdp'],
      '#button_type' => 'secondary',
      '#weight' => 5,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitTestIdp(array &$form, FormStateInterface $form_state): void {
    $form_state->setRebuild(TRUE);
    $form_state->set('simplesamlphp_sp_test_preview_url', NULL);
    $form_state->set('simplesamlphp_sp_test_preview_description', NULL);

    $sp_name = trim((string) $form_state->getValue('sp_name')) ?: 'default-sp';
    $base_dir = trim((string) $form_state->getValue('simplesamlphp_base_dir'));

    $previous_env = getenv('SIMPLESAMLPHP_INSTALL_DIR');
    $had_env = $previous_env !== FALSE;
    $previous_env = $had_env ? (string) $previous_env : '';

    try {
      if ($base_dir !== '') {
        $this->setSimpleSamlInstallDir($base_dir);
        $this->simpleSamlAuthClientFactory->create($sp_name);
      }
      else {
        $this->clearSimpleSamlInstallDir();
        if (!class_exists(Simple::class)) {
          throw new \RuntimeException((string) $this->t('SimpleSAMLphp library is not available. Provide a base directory or install it via Composer.'));
        }
      }

      $simple = new Simple($sp_name);
      $simple->getAuthSource();

      if ($preview = $this->buildIdpMetadataPreview($simple)) {
        $form_state->set('simplesamlphp_sp_test_preview_url', $preview['src']);
        $form_state->set('simplesamlphp_sp_test_preview_description', $preview['description']);
        $this->messenger()->addStatus($this->t('SAML configuration appears valid. The Identity Provider metadata is embedded below.'));
      }
      else {
        $this->messenger()->addStatus($this->t('SAML configuration appears valid, but the Identity Provider metadata could not be located.'));
      }
    }
    catch (\Throwable $throwable) {
      $this->messenger()->addError($this->t('SAML configuration test failed: @message', ['@message' => $throwable->getMessage()]));
    }
    finally {
      if ($had_env) {
        $this->setSimpleSamlInstallDir($previous_env);
      }
      else {
        $this->clearSimpleSamlInstallDir();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $lock_external = (bool) $form_state->getValue('lock_external_user_fields');
    $allow_native_login = (bool) $form_state->getValue('allow_native_login_for_external_users');
    $exempt_user_ids = $this->sanitizeExemptUserIds((array) $form_state->getValue('exempt_user_ids'));
    $log_debug_information = (bool) $form_state->getValue('log_debug_information');

    $this->config('simplesamlphp_sp.settings')
      ->set('activate', $form_state->getValue('activate'))
      ->set('simplesamlphp_base_dir', $form_state->getValue('simplesamlphp_base_dir'))
      ->set('sp_name', $form_state->getValue('sp_name'))
      ->set('saml_login_path', $form_state->getValue('saml_login_path'))
      ->set('unique_id_attribute', $form_state->getValue('unique_id_attribute'))
      ->set('username_attribute', $form_state->getValue('username_attribute'))
      ->set('email_attribute', $form_state->getValue('email_attribute'))
      ->set('blocked_roles', $this->sanitizeBlockedRoles((array) $form_state->getValue('blocked_roles')))
      ->set('lock_external_user_fields', $lock_external)
      ->set('allow_native_login_for_external_users', $allow_native_login)
      ->set('log_debug_information', $log_debug_information)
      ->set('exempt_user_ids', $exempt_user_ids)
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Normalizes the blocked_roles value before persisting it.
   *
   * @param array $selected_roles
   *   Selected role identifiers from the form submission.
   *
   * @return array
   *   Sanitized list of role IDs to store in configuration.
   */
  protected function sanitizeBlockedRoles(array $selected_roles): array {
    $none_option_key = static::NONE_OPTION_KEY;
    $selected_roles = array_values(array_filter($selected_roles));

    if (in_array($none_option_key, $selected_roles, TRUE)) {
      return [];
    }

    return $selected_roles;
  }

  /**
   * Normalizes the exempt user list before saving configuration.
   */
  protected function sanitizeExemptUserIds(array $selected_users): array {
    $ids = [];
    foreach ($selected_users as $value) {
      if (is_array($value) && isset($value['target_id'])) {
        $ids[] = (int) $value['target_id'];
      }
      elseif (is_numeric($value)) {
        $ids[] = (int) $value;
      }
    }

    $ids[] = 1;
    $ids = array_unique(array_filter($ids, static fn($value) => $value > 0));
    sort($ids, SORT_NUMERIC);

    return array_values($ids);
  }

  /**
   * Builds an IdP metadata preview payload for the iframe.
   */
  private function buildIdpMetadataPreview(Simple $simple): ?array {
    $auth_source = $simple->getAuthSource();
    if (!$auth_source instanceof SP) {
      return NULL;
    }

    $sp_metadata = $auth_source->getMetadata();
    $idp_entity_id = $sp_metadata->getOptionalString('idp', NULL);
    if (!is_string($idp_entity_id) || $idp_entity_id === '') {
      return NULL;
    }

    try {
      $idp_metadata = $auth_source->getIdPMetadata($idp_entity_id);
    }
    catch (\Throwable $throwable) {
      return NULL;
    }

    $metadata_array = $idp_metadata->toArray();
    $metadata_json = json_encode($metadata_array, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    if ($metadata_json === FALSE) {
      return NULL;
    }

    $data_url = 'data:text/plain;charset=utf-8,' . rawurlencode($metadata_json);

    $metadata_url = $idp_metadata->getOptionalString('metadataURL', NULL);
    $description = $metadata_url && is_string($metadata_url)
      ? $this->t('Displaying an IdP metadata snapshot generated from @url.', ['@url' => $metadata_url])
      : $this->t('Displaying a JSON snapshot generated from the IdP metadata configuration.');

    return [
      'src' => $data_url,
      'description' => (string) $description,
    ];
  }

  /**
   * Sets the SimpleSAML installation directory environment variable.
   */
  private function setSimpleSamlInstallDir(string $value): void {
    putenv("SIMPLESAMLPHP_INSTALL_DIR={$value}");
    $_ENV['SIMPLESAMLPHP_INSTALL_DIR'] = $value;
    $_SERVER['SIMPLESAMLPHP_INSTALL_DIR'] = $value;
  }

  /**
   * Removes the SimpleSAML installation directory environment variable.
   */
  private function clearSimpleSamlInstallDir(): void {
    putenv('SIMPLESAMLPHP_INSTALL_DIR');
    unset($_ENV['SIMPLESAMLPHP_INSTALL_DIR'], $_SERVER['SIMPLESAMLPHP_INSTALL_DIR']);
  }

}
