<?php

namespace Drupal\Tests\simplesamlphp_sp\Functional;

use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper;
use Drupal\user\Entity\User;
use Drupal\user\UserInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Tests the configurable login path for SimpleSAMLphp SP.
 *
 * @group simplesamlphp_sp
 */
class SimpleSamlSpTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'simplesamlphp_sp',
    'externalauth',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer simplesamlphp sp configuration',
      'administer users',
      'access administration pages',
    ]);

    // Set a custom login path.
    $this->config('simplesamlphp_sp.settings')
      ->set('saml_login_path', '/my-custom-saml-login')
      ->save();

    // Rebuild routes.
    $this->rebuildAll();
  }

  /**
   * Tests that an authenticated user is redirected.
   */
  public function testAuthenticatedUserRedirect(): void {
    $assert = $this->assertSession();
    $this->drupalLogin($this->adminUser);

    // Test the path for an authenticated user. They should be redirected.
    $this->drupalGet('/my-custom-saml-login');
    $assert->pageTextContains($this->adminUser->getAccountName());
    $assert->statusCodeEquals(200);
  }

  /**
   * Ensures only users with the module permission can access the settings UI.
   */
  public function testSettingsFormPermission(): void {
    $assert = $this->assertSession();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/people/simplesamlphp-sp');
    $assert->statusCodeEquals(200);

    $this->drupalLogout();

    $restricted_user = $this->drupalCreateUser([
      'access administration pages',
      'administer site configuration',
    ]);
    $this->drupalLogin($restricted_user);
    $this->drupalGet('/admin/config/people/simplesamlphp-sp');
    $this->assertSession()->statusCodeEquals(Response::HTTP_FORBIDDEN);
  }

  /**
   * Ensures SAML authentication is skipped when the module is inactive.
   */
  public function testSamlInactiveRedirectsToLogin(): void {
    $assert = $this->assertSession();

    // Ensure the feature flag is disabled.
    $this->config('simplesamlphp_sp.settings')
      ->set('activate', FALSE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');

    // When disabled, the controller should redirect to Drupal's login page.
    $assert->statusCodeEquals(200);
    $assert->addressEquals($this->buildUrl('/user/login'));
  }

  /**
   * The Authentication process test.
   */
  public function testAuthenticationProcess(): void {
    $assert = $this->assertSession();
    // Anonymous attempt.
    $this->drupalGet('/my-custom-saml-login');
    $assert->statusCodeNotEquals(404);

    $this->assertAuthCookie(FALSE, 'Drupal authentication cookie should not be set for anonymous users on the login path.');

    // Simulate the situation where the SSO is successfully.
    $this->configureSamlEnvironment();

    // Now, when we visit the login path, the test module should simulate a
    // successful SAML authentication, and Drupal should set a session cookie.
    $this->drupalGet('/my-custom-saml-login');

    $this->assertAuthCookie(TRUE, 'Drupal authentication cookie was set after successful SSO simulation.');
    $assert->pageTextContains('testuser');

    // Test the logout process.
    $this->drupalGet('/user/logout/confirm');
    $this->submitForm([], 'Log out', 'user-logout-confirm');
    $this->assertAuthCookie(FALSE, 'Drupal authentication cookie should not be set after logout.');
  }

  /**
   * Ensures administrator accounts cannot log in via SAML.
   */
  public function testAdministratorLoginDenied(): void {
    $this->configureSamlEnvironment();

    $this->createAdminRole('administrator', 'Administrator');

    $admin_account = $this->drupalCreateUser([], 'testuser', FALSE, ['mail' => 'testuser@example.com']);
    $admin_account->addRole('administrator');
    $admin_account->save();

    $this->drupalGet('/my-custom-saml-login');
    $this->assertDeniedRedirect('restricted_role', 'Administrator accounts should be denied SAML login.');
  }

  /**
   * Ensures custom roles can be denied via configuration.
   */
  public function testCustomRoleLoginDenied(): void {
    $this->configureSamlEnvironment();

    $denied_role = $this->drupalCreateRole([], 'saml_denied', 'SAML denied');
    $this->config('simplesamlphp_sp.settings')
      ->set('blocked_roles', [$denied_role])
      ->save();

    $user = $this->drupalCreateUser([], 'roleblocked', FALSE, ['mail' => 'testuser@example.com']);
    $user->addRole($denied_role);
    $user->save();

    $this->drupalGet('/my-custom-saml-login');
    $this->assertDeniedRedirect('restricted_role', 'Users with configured restricted roles should be denied SAML login.');
  }

  /**
   * Ensures temporary bypass allows blocked roles to authenticate via SAML.
   */
  public function testTemporaryBypassAllowsRestrictedRole(): void {
    $this->configureSamlEnvironment();

    $this->createAdminRole('administrator', 'Administrator');
    $user = $this->drupalCreateUser([], 'tempbypass', FALSE, ['mail' => 'testuser@example.com']);
    $user->addRole('administrator');
    $user->save();

    $this->drupalGet('/my-custom-saml-login');
    $this->assertDeniedRedirect('restricted_role', 'Administrator accounts should be denied before bypass is applied.');

    /** @var \Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper $helper */
    $helper = $this->container->get('simplesamlphp_sp.account_helper');
    $this->assertInstanceOf(SimpleSamlAccountHelper::class, $helper);
    $helper->setTemporaryBypassMinutes($user, 30);

    $this->drupalGet('/my-custom-saml-login');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertAuthCookie(TRUE, 'Administrator accounts should authenticate while bypass is active.');
    $this->assertSession()->pageTextContains($user->getAccountName());

    $this->drupalLogout();

    $time = $this->container->get('datetime.time')->getRequestTime();
    $this->container->get('user.data')->set('simplesamlphp_sp', $user->id(), 'temporary_bypass_expires', $time - 60);

    $this->drupalGet('/my-custom-saml-login');
    $this->assertDeniedRedirect('restricted_role', 'Administrator accounts should be denied once the bypass expires.');

    $this->assertNull($this->container->get('user.data')->get('simplesamlphp_sp', $user->id(), 'temporary_bypass_expires'), 'Expired bypass values should be cleared automatically.');
  }

  /**
   * Ensures user 1 can never authenticate via SAML.
   */
  public function testSuperUserLoginDenied(): void {
    $this->configureSamlEnvironment();

    $super_user = User::load(1);
    $this->assertNotNull($super_user, 'User 1 exists.');
    $super_user->setEmail('testuser@example.com');
    $super_user->save();

    $this->drupalGet('/my-custom-saml-login');
    $this->assertDeniedRedirect('user_1', 'Super administrator account should be denied SAML login.');
  }

  /**
   * Ensures selecting "None" allows all roles to authenticate.
   */
  public function testNoneOptionAllowsAllRoles(): void {
    $this->configureSamlEnvironment();

    $this->config('simplesamlphp_sp.settings')
      ->set('blocked_roles', [])
      ->save();

    $this->createAdminRole('administrator', 'Administrator');
    $admin_account = $this->drupalCreateUser([], 'adminallowed', FALSE, ['mail' => 'testuser@example.com']);
    $admin_account->addRole('administrator');
    $admin_account->save();

    $this->drupalGet('/my-custom-saml-login');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertAuthCookie(TRUE, 'Administrator should authenticate successfully when no roles are blocked.');
    $this->assertSession()->pageTextContains('adminallowed');
  }

  /**
   * Ensures locked external accounts cannot change core credential fields.
   */
  public function testLockedExternalAccountsCannotChangeCredentials(): void {
    $this->configureSamlEnvironment();
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', TRUE)
      ->set('allow_native_login_for_external_users', FALSE)
      ->save();

    // Trigger the mock SAML login to provision the account.
    $this->drupalGet('/my-custom-saml-login');

    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);

    $original_name = $account->getAccountName();
    $original_mail = $account->getEmail();
    $original_password_hash = $account->getPassword();

    $account->setUsername('alteredname');
    $account->setEmail('changed@example.com');
    $account->setPassword('new-password');
    $account->save();

    $reloaded = User::load($account->id());
    $this->assertSame($original_name, $reloaded->getAccountName(), 'Username remains locked for externally managed accounts.');
    $this->assertSame($original_mail, $reloaded->getEmail(), 'Email remains locked for externally managed accounts.');
    $this->assertSame($original_password_hash, $reloaded->getPassword(), 'Password hash remains locked for externally managed accounts.');
  }

  /**
   * Ensures credential fields are hidden on user edit forms when locked.
   */
  public function testLockedExternalAccountEditFormHidesCredentialFields(): void {
    $this->configureSamlEnvironment();
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', TRUE)
      ->set('allow_native_login_for_external_users', FALSE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');
    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);

    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);

    $this->drupalGet('/user/' . $account->id() . '/edit');
    $assert = $this->assertSession();
    $assert->elementNotExists('css', '#edit-name');
    $assert->elementNotExists('css', '#edit-mail');
    $assert->elementNotExists('css', '#edit-account-pass');
    $assert->elementNotExists('css', '#edit-current-pass');
  }

  /**
   * Ensures Drupal-native login is blocked when the toggle disallows it.
   */
  public function testNativeLoginBlockedWhenDisallowed(): void {
    $this->configureSamlEnvironment();

    // Provision the account without restrictions so we can assign a password.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', TRUE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');
    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);

    $account->setPassword('native-pass');
    $account->save();

    $this->drupalLogout();

    // Disallow native login while keeping credential locking disabled to
    // confirm the restriction relies solely on the allow toggle.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', FALSE)
      ->save();
    $this->rebuildAll();

    $this->drupalGet('/user/login');
    $this->submitForm(['name' => 'testuser', 'pass' => 'native-pass'], 'Log in');

    $assert = $this->assertSession();
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Unrecognized username or password.');
    $this->assertAuthCookie(FALSE, 'SAML accounts should not authenticate via Drupal native login when restricted.');

    $login_endpoint = Url::fromRoute('user.login.http', [], [
      'absolute' => TRUE,
      'query' => ['_format' => 'json'],
    ])->toString();
    $response = $this->postJson($login_endpoint, ['name' => 'testuser', 'pass' => 'native-pass']);
    $this->assertSame(400, $response->getStatusCode());
    $body = json_decode($response->getContent(), TRUE);
    $this->assertSame('Sorry, unrecognized username or password.', $body['message']);
  }

  /**
   * Ensures native login stays available when explicitly permitted.
   */
  public function testNativeLoginAllowedWhenPermitted(): void {
    $this->configureSamlEnvironment();

    // Provision the account and set a known password.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', TRUE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');
    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);

    $account->setPassword('native-pass');
    $account->save();

    $this->drupalLogout();

    // Lock the account credentials but leave native login permitted to confirm
    // the allow toggle alone governs access.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', TRUE)
      ->set('allow_native_login_for_external_users', TRUE)
      ->save();
    $this->rebuildAll();

    $this->drupalGet('/user/login');
    $this->submitForm(['name' => 'testuser', 'pass' => 'native-pass'], 'Log in');
    $assert = $this->assertSession();
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Member for');
    $this->assertAuthCookie(TRUE, 'SAML accounts should authenticate via Drupal native login when permitted.');

    $this->drupalLogout();

    // JSON login requests should also succeed.
    $login_endpoint = Url::fromRoute('user.login.http', [], [
      'absolute' => TRUE,
      'query' => ['_format' => 'json'],
    ])->toString();
    $response = $this->postJson($login_endpoint, ['name' => 'testuser', 'pass' => 'native-pass']);
    $this->assertSame(200, $response->getStatusCode());

    // Password reset flow should proceed without exposing restriction messages.
    $this->drupalLogout();
    $this->drupalGet('/user/password');
    $this->submitForm(['name' => 'testuser'], 'Submit');
    $assert->pageTextContains('If testuser is a valid account, an email will be sent with instructions to reset your password.');

    $password_reset_endpoint = Url::fromRoute('user.pass.http', [], [
      'absolute' => TRUE,
      'query' => ['_format' => 'json'],
    ])->toString();
    $jsonReset = $this->postJson($password_reset_endpoint, ['name' => 'testuser']);
    $this->assertSame(200, $jsonReset->getStatusCode());
  }

  /**
   * Ensures password reset routes are blocked when native login is disallowed.
   */
  public function testNativePasswordResetBlockedWhenDisallowed(): void {
    $this->configureSamlEnvironment();

    // Allow edits long enough to seed a known password.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', TRUE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');
    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);

    $account->setPassword('native-pass');
    $account->save();

    $this->drupalLogout();

    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', FALSE)
      ->save();
    $this->rebuildAll();

    $account = User::load($account->id());

    $assert = $this->assertSession();

    $this->drupalGet('/user/password');
    $this->submitForm(['name' => 'testuser'], 'Submit');
    $assert->pageTextContains('If testuser is a valid account, an email will be sent with instructions to reset your password.');

    $password_reset_endpoint = Url::fromRoute('user.pass.http', [], [
      'absolute' => TRUE,
      'query' => ['_format' => 'json'],
    ])->toString();
    $jsonReset = $this->postJson($password_reset_endpoint, ['name' => 'testuser']);
    $this->assertSame(400, $jsonReset->getStatusCode());
    $jsonBody = json_decode($jsonReset->getContent(), TRUE);
    $this->assertSame('Unable to send email. Contact the site administrator if the problem persists.', $jsonBody['message']);

    $reset_url = user_pass_reset_url($account);
    $this->drupalGet($reset_url);
    $assert->pageTextContains('Password reset links cannot be used with accounts managed by Single Sign-On.');
    $assert->elementNotExists('css', '#edit-actions-submit');

    $parsed = parse_url($reset_url);
    $segments = explode('/', trim($parsed['path'] ?? '', '/'));
    $timestamp = $segments[3] ?? NULL;
    $hash = $segments[4] ?? NULL;
    $this->assertNotEmpty($timestamp);
    $this->assertNotEmpty($hash);

    $login_url = Url::fromRoute('user.reset.login', [
      'uid' => $account->id(),
      'timestamp' => $timestamp,
      'hash' => $hash,
    ])->toString();

    $this->getSession()->getDriver()->getClient()->request('POST', $login_url);
    $assert->addressEquals($this->buildUrl('/user/login'));
    $assert->pageTextContains('Password reset links cannot be used');
    $this->assertAuthCookie(FALSE, 'SAML accounts should not authenticate via one-time login when restricted.');
  }

  /**
   * Ensures exempt users bypass credential locking and native login blocks.
   */
  public function testExemptUsersBypassRestrictions(): void {
    $this->configureSamlEnvironment();

    // Provision the SAML account and give it a known password.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', FALSE)
      ->set('allow_native_login_for_external_users', TRUE)
      ->save();

    $this->drupalGet('/my-custom-saml-login');
    $account = $this->loadUserByName('testuser');
    $this->assertNotNull($account);
    $account->setPassword('native-pass');
    $account->save();

    // Restrict native login globally but exempt the account.
    $this->config('simplesamlphp_sp.settings')
      ->set('lock_external_user_fields', TRUE)
      ->set('allow_native_login_for_external_users', FALSE)
      ->set('exempt_user_ids', [1, (int) $account->id()])
      ->save();
    $this->rebuildAll();

    // Exempt users should still have credential fields on the edit form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/user/' . $account->id() . '/edit');
    $assert = $this->assertSession();
    $assert->elementExists('css', '#edit-pass-pass1');
    $assert->elementExists('css', '#edit-mail');

    $new_password = 'new-password';
    $this->submitForm([
      'Email address' => 'exemptuser@example.com',
      'Current password' => 'native-pass',
      'Password' => $new_password,
      'Confirm password' => $new_password,
    ], 'Save');
    $account = User::load($account->id());
    $this->assertSame('exemptuser@example.com', $account->getEmail());
    $this->drupalLogout();

    // Exempt users should be able to log in via Drupal native login.
    $this->drupalGet('/user/login');
    $this->submitForm(['name' => $account->getAccountName(), 'pass' => $new_password], 'Log in');
    $assert = $this->assertSession();
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Member for');
    $this->assertAuthCookie(TRUE, 'Exempt SAML users should authenticate via Drupal native login.');
  }

  /**
   * Asserts whether a Drupal authentication cookie exists.
   *
   * @param bool $should_exist
   *   Whether the cookie should exist or not.
   * @param string $message
   *   The message to display for the assertion.
   */
  protected function assertAuthCookie(bool $should_exist, string $message = ''): void {
    $cookies = $this->getSession()->getDriver()->getClient()->getCookieJar()->all();
    $auth_cookie_found = FALSE;
    foreach ($cookies as $cookie) {
      if (preg_match('/^S?SESS/', $cookie->getName())) {
        $auth_cookie_found = TRUE;
        break;
      }
    }

    if ($should_exist) {
      $this->assertTrue($auth_cookie_found, $message);
    }
    else {
      $this->assertFalse($auth_cookie_found, $message);
    }
  }

  /**
   * Sends a JSON POST request and returns the raw response.
   */
  protected function postJson(string $url, array $payload): Response {
    $client = $this->getSession()->getDriver()->getClient();
    $target = str_starts_with($url, 'http') ? $url : $this->buildUrl($url);
    $client->request('POST', $target, [], [], [
      'CONTENT_TYPE' => 'application/json',
      'HTTP_ACCEPT' => 'application/json',
    ], json_encode($payload));
    $raw = $client->getResponse();

    return new Response(
      $raw->getContent(),
      $raw->getStatusCode(),
      $raw->getHeaders()
    );
  }

  /**
   * Asserts that the current page is the denial endpoint with the given reason.
   */
  protected function assertDeniedRedirect(string $expected_reason, string $cookie_message): void {
    $this->assertSession()->statusCodeEquals(403);
    $current_url = $this->getSession()->getCurrentUrl();
    $expected_path = parse_url($this->buildUrl('/saml/login-denied'), PHP_URL_PATH);
    $this->assertSame($expected_path, parse_url($current_url, PHP_URL_PATH));
    $query = [];
    parse_str((string) parse_url($current_url, PHP_URL_QUERY), $query);
    $this->assertSame($expected_reason, $query['reason'] ?? NULL);
    $this->assertAuthCookie(FALSE, $cookie_message);
  }

  /**
   * Prepares the mocked SimpleSAML environment for functional tests.
   */
  protected function configureSamlEnvironment(): void {
    if (!$this->container->get('module_handler')->moduleExists('simplesamlphp_sp_test')) {
      $this->container->get('module_installer')->install(['simplesamlphp_sp_test']);
      $this->rebuildAll();
    }

    $this->config('simplesamlphp_sp.settings')
      ->set('activate', TRUE)
      ->set('unique_id_attribute', 'uid')
      ->set('username_attribute', 'name')
      ->set('email_attribute', 'mail')
      ->set('blocked_roles', ['administrator'])
      ->set('exempt_user_ids', [1])
      ->save();
  }

  /**
   * Loads a user account by its username.
   */
  protected function loadUserByName(string $username): ?UserInterface {
    $accounts = $this->container->get('entity_type.manager')->getStorage('user')->loadByProperties(['name' => $username]);
    if (!$accounts) {
      return NULL;
    }
    $account = reset($accounts);
    return $account instanceof UserInterface ? $account : NULL;
  }

}
