# SimpleSAMLphp SP

SimpleSAMLphp SP provides a Drupal 11 integration layer for sites that authenticate users through a SimpleSAMLphp service provider. It bridges Drupal's account system with an external Identity Provider (IdP), manages SSO entry points, and enforces policies for SAML-provisioned accounts.

## Features
- Adds a configurable SAML login route that proxies requests through a SimpleSAMLphp service provider.
- Links IdP users to Drupal accounts using the External Auth module, creating or connecting accounts automatically.
- Protects admin roles and the super admin (UID 1) from being taken over by external accounts.
- Locks Drupal-native credential fields and login interfaces for SAML-managed accounts when desired, while honoring configurable exemptions.
- Initiates coordinated logout between Drupal and the IdP using SimpleSAMLphp sessions.
- Surface-friendly denial handling that redirects rejected logins to a dedicated route with contextual messaging.

## Requirements
- PHP 8.1 or later.
- Drupal core ^11.0.
- [External Auth](https://www.drupal.org/project/externalauth) module (^2.0).
- SimpleSAMLphp 2.5 library available either via Composer (`simplesamlphp/simplesamlphp:dev-simplesamlphp-2.5`) or an on-disk installation referenced by configuration.
- A SimpleSAMLphp stand-alone service provider instance installed and configured (authsources, IdP metadata, certificates, etc.) that this module can bootstrap.
- Ability to reach a SimpleSAMLphp service provider configuration (typically via `config/` in your SimpleSAMLphp install).

## Installation
1. Install this module through the normal Drupal module installation process or install it via Composer.
2. Ensure the SimpleSAMLphp library is installed. If not managed by Composer, set the `SIMPLESAMLPHP_INSTALL_DIR` environment variable or note the full path to the installation directory.
3. Enable dependencies and the module, for example:
   ```sh
   drush en externalauth simplesamlphp_sp
   ```
4. Clear Drupal caches so the dynamic SAML login route is registered:
   ```sh
   drush cr
   ```

## Configuration
1. Navigate to **Configuration → People → SimpleSAMLphp SP settings** (`/admin/config/people/simplesamlphp-sp`).
2. Provide the SimpleSAMLphp base directory (if not using the environment variable) and the service provider name exactly as defined in `config/authsources.php` (defaults to `default-sp`).
3. Define the public **SAML login path**. The module dynamically exposes this path and routes requests to the IdP. The default is `/saml/login`.
4. Map IdP attributes to Drupal fields:
   - Unique identifier attribute (used to look up/create the account).
   - Username attribute (falls back to the unique identifier when empty).
   - Email attribute (required for account linking and provisioning).
5. Optionally restrict which Drupal roles may authenticate via SAML. Leave the selection empty (or choose the “None” option) to allow all roles.
6. Decide whether to lock username/email/password fields for accounts provisioned by SAML. When locked, native Drupal login, one-time login links, and password resets can also be disabled for those accounts.
7. Specify user IDs that are exempt from credential locking and native login restrictions. User 1 is always exempt.
8. Save the configuration and test the SSO flow by visiting the configured SAML login path.

### SimpleSAMLphp Library Loading
- When `SIMPLESAMLPHP_INSTALL_DIR` is set in the environment, it takes precedence and must point to the SimpleSAMLphp root directory (containing `config/`, `modules/`, and `www/`).
- Otherwise, the module uses the configured base directory. The path must contain `lib/_autoload.php`; a runtime exception is thrown if the file cannot be located.
- If the library is already available through Composer autoloading, the base directory can be left empty.

## Account Linking & Restrictions
- `SimpleSamlSpAuth` relies on the External Auth service to load or create Drupal accounts by the IdP’s unique identifier.
- Existing Drupal accounts with matching email addresses are linked automatically (subject to policy checks).
- User 1 and accounts with restricted roles are blocked from SAML login; denials produce a redirect to `/saml/login-denied` with contextual messaging.
- `SimpleSamlAccountHelper` marks accounts created by SAML and exposes exemption utilities used by hooks and event subscribers.

## Native Login Controls
- `SimplesamlphpSpHooks` adds validation to Drupal login and password reset forms, preventing SAML-managed accounts from authenticating natively when credential locking is enabled.
- `SimpleSamlNativeLoginSubscriber` intercepts JSON login/password endpoints and one-time login links for SAML accounts, returning friendly error responses or redirects.

## Extending Post-Login Behaviour
- Implement `hook_simplesamlphp_sp_post_login()` in another module to react immediately after a SAML login succeeds (before the final redirect).
- The hook receives the authenticated `UserInterface` object, the raw attribute array returned by the IdP, and the resolved `$authname`, `$username`, and `$email` values.
- Use the hook to synchronize profile data, adjust roles, or persist custom metadata; remember to call `$account->save()` if you modify the user entity.
- Example:
  ```php
  /**
   * Implements hook_simplesamlphp_sp_post_login().
   */
  function my_module_simplesamlphp_sp_post_login(\Drupal\user\UserInterface $account, array $attributes, string $authname, string $username, string $email): void {
    if (!empty($attributes['department'][0])) {
      $account->set('field_department', $attributes['department'][0]);
      $account->save();
    }
  }
  ```

## Logout Behaviour
- On Drupal logout, the module clears the local session and, if SSO is active, instructs SimpleSAMLphp to perform an IdP logout. The IdP can redirect back to Drupal’s front page or a supplied return URL.

## Troubleshooting
- **“SimpleSAMLphp base dir not found”** – verify `SIMPLESAMLPHP_INSTALL_DIR` or the configured base path and ensure the directory exists within the Drupal container or filesystem.
- **No attributes returned** – confirm the service provider configuration on the IdP and that the requested attributes match names provided by the IdP response.
- **403 Access denied after login** – the account likely matches a restricted role or is user 1. Adjust the “Roles denied SAML login” list or exempt the account.
- **Native login unexpectedly allowed** – ensure credential locking is enabled and that the user is not in the exempt list.

## Development Notes
- Services are defined in `simplesamlphp_sp.services.yml` and use PSR-4 classes under `Drupal\simplesamlphp_sp\`.
- The dynamic SAML login route is built by `SimpleSamlSpRouteBuilder`, so configuration changes may require a cache rebuild.
- Hooks live in `SimplesamlphpSpHooks`, exposed through a service fetched by the procedural `simplesamlphp_sp.module` wrapper for backward-compatible hook invocation.

## License
This module is distributed under the same license as Drupal core (GPL-2.0-or-later).
