<?php

namespace Drupal\simplesamlphp_sp\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Session\AnonymousUserSession;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper;
use Drupal\simplesamlphp_sp\Service\SimpleSamlManager;
use Drupal\user\UserInterface;

/**
 * Hook implementations for the SimpleSAMLphp SP module.
 */
class SimplesamlphpSpHooks {
  use StringTranslationTrait;

  /**
   * The SimpleSAMLphp SP manager.
   */
  protected SimpleSamlManager $samlManager;

  /**
   * The config factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger channel for this module.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Entity type manager service.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Helper for identifying SAML-managed accounts.
   */
  protected SimpleSamlAccountHelper $accountHelper;

  /**
   * Current user service.
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Date formatter service.
   */
  protected DateFormatterInterface $dateFormatter;

  public function __construct(SimpleSamlManager $samlManager, ConfigFactoryInterface $configFactory, LoggerChannelInterface $logger, EntityTypeManagerInterface $entityTypeManager, SimpleSamlAccountHelper $accountHelper, AccountProxyInterface $currentUser, DateFormatterInterface $dateFormatter) {
    $this->samlManager = $samlManager;
    $this->configFactory = $configFactory;
    $this->logger = $logger;
    $this->entityTypeManager = $entityTypeManager;
    $this->accountHelper = $accountHelper;
    $this->currentUser = $currentUser;
    $this->dateFormatter = $dateFormatter;
  }

  /**
   * Implements hook_user_logout().
   */
  #[Hook('user_logout')]
  public function userLogout(AccountInterface $account): void {
    $user = $this->currentUser;
    // Keep the user id before forcing a logout.
    $user_id = $user->id();

    // Drupal normally invalidates the session before this hook fires, but we
    // repeat the step to avoid any ambiguity if the IdP redirect cuts the
    // logout flow short.
    if ($user->isAuthenticated()) {
      $this->logger->info('Invalidating session before initiating SAML logout.');
      $request = \Drupal::request();
      $session = $request->getSession();
      $session->invalidate();
      $user->setAccount(new AnonymousUserSession());
    }

    // When SSO feature is inactive, do nothing.
    if (!$this->samlManager->isActivated()) {
      $this->logger->info('SimpleSAMLphp SSO is disabled; skipping SAML logout.');
      return;
    }
    // Start the SSO logout process to IDP.
    $this->logger->info('Initiating SAML logout for user @uid.', ['@uid' => $user_id]);
    $this->samlManager->logout();
  }

  /**
   * Implements hook_user_presave().
   */
  #[Hook('user_presave')]
  public function userPresave(UserInterface $account): void {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    if (!$config->get('lock_external_user_fields')) {
      return;
    }

    if ($account->isNew()) {
      return;
    }

    if (!$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $original_account = $account->original ?? $this->entityTypeManager->getStorage('user')->load($account->id());
    if (!$original_account instanceof UserInterface) {
      return;
    }

    $blocked_fields = [];

    if ($account->getAccountName() !== $original_account->getAccountName()) {
      $account->setUsername($original_account->getAccountName());
      $blocked_fields[] = 'username';
    }

    if ($account->getEmail() !== $original_account->getEmail()) {
      $account->setEmail($original_account->getEmail());
      $blocked_fields[] = 'email';
    }

    $current_password = $account->getPassword();
    $original_password = $original_account->getPassword();
    if ($current_password !== $original_password) {
      $account->get('pass')->value = $original_password;
      $blocked_fields[] = 'password';
    }

    if ($blocked_fields) {
      $this->logger->info('Prevented updates to @fields for externally managed user account @uid.', [
        '@fields' => implode(', ', $blocked_fields),
        '@uid' => $account->id(),
      ]);
    }
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    $lockCredentials = (bool) $config->get('lock_external_user_fields');
    $restrictNativeLogin = $this->accountHelper->shouldRestrictNativeLogin();

    $form_object = $form_state->getFormObject();
    if ($form_object instanceof EntityFormInterface) {
      $entity = $form_object->getEntity();
      if ($entity instanceof UserInterface) {
        // Hide user credential fields for SAML-managed accounts.
        if ($lockCredentials && !$entity->isNew() && $this->accountHelper->isSamlAccount($entity) && !$this->accountHelper->isExempt($entity)) {
          $this->restrictCredentialFields($form);
        }
      }
    }

    switch ($form_id) {
      case 'user_form':
        if ($form_object instanceof EntityFormInterface) {
          $entity = $form_object->getEntity();
          if ($entity instanceof UserInterface && !$entity->isNew() && (int) $entity->id() !== 1) {
            if ($this->currentUser->hasPermission('administer simplesamlphp sp configuration')) {
              $this->addTemporaryBypassField($form, $entity);

              if (isset($form['actions']['submit']) && is_array($form['actions']['submit'])) {
                $form['actions']['submit']['#submit'][] = [$this, 'submitUserEntityForm'];
              }
              else {
                $form['#submit'][] = [$this, 'submitUserEntityForm'];
              }
            }
          }
        }
        break;

      case 'user_login_form':
      case 'user_login_block':
        if ($restrictNativeLogin) {
          $form['#validate'][] = [$this, 'validateNativeLoginRestriction'];
        }
        break;

      case 'user_pass_reset':
        if ($restrictNativeLogin) {
          $this->alterPasswordResetForm($form, $form_state);
        }
        break;
    }
  }

  /**
   * Hides credential-related elements on the user account edit form.
   */
  protected function restrictCredentialFields(array &$form): void {
    $targets = [
      ['account', 'name'],
      ['account', 'mail'],
      ['account', 'pass'],
      ['account', 'current_pass'],
    ];

    foreach ($targets as $path) {
      $this->setElementAccess($form, $path, FALSE);
    }
  }

  /**
   * Adds the temporary SAML bypass configuration element to user forms.
   */
  protected function addTemporaryBypassField(array &$form, UserInterface $account): void {
    if (!isset($form['account']) || !is_array($form['account'])) {
      return;
    }

    $remaining = $this->accountHelper->getRemainingTemporaryBypassMinutes($account);
    $expires = $this->accountHelper->getTemporaryBypassExpiration($account);

    $description = $this->t('Temporarily allow this user to bypass the "Roles denied SAML login" setting for the specified number of minutes (maximum 1440). 0 means no bypass.');
    if ($remaining > 0 && $expires > 0) {
      $formatted_expiration = $this->dateFormatter->format($expires, 'long');
      $description .= '<br>' . $this->t('Current allowance expires on <strong>@date</strong>.', ['@date' => $formatted_expiration]);
    }

    $form['account']['simplesamlphp_sp_temp_bypass'] = [
      '#type' => 'number',
      '#title' => $this->t('Temporary SAML allowance (minutes)'),
      '#description' => $description,
      '#default_value' => $remaining,
      '#min' => 0,
      '#max' => 1440,
      '#step' => 1,
      '#weight' => 90,
      '#element_validate' => [[$this, 'validateTemporaryBypassField']],
    ];
  }

  /**
   * Validates the temporary SAML bypass field value.
   */
  public function validateTemporaryBypassField(array &$element, FormStateInterface $form_state, array &$complete_form): void {
    $raw = $form_state->getValue($element['#parents'] ?? ['simplesamlphp_sp_temp_bypass']);
    if ($raw === NULL || $raw === '') {
      $raw = 0;
    }

    if (!is_numeric($raw)) {
      $form_state->setError($element, $this->t('Enter a numeric value between 0 and 1440.'));
      return;
    }

    $value = (int) $raw;
    if ($value < 0 || $value > 1440) {
      $form_state->setError($element, $this->t('Temporary SAML allowance must be between 0 and 1440 minutes.'));
      return;
    }

    $form_state->setValueForElement($element, $value);
  }

  /**
   * Persists temporary bypass settings after user form submission.
   */
  public function submitUserEntityForm(array &$form, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();
    if (!$form_object instanceof EntityFormInterface) {
      return;
    }

    $entity = $form_object->getEntity();
    if (!$entity instanceof UserInterface || $entity->isNew()) {
      return;
    }

    $minute = $form_state->getValue('simplesamlphp_sp_temp_bypass');
    if ($minute === NULL) {
      $minute = $form_state->getValue(['account', 'simplesamlphp_sp_temp_bypass']);
    }
    if ($minute === NULL) {
      $element = $form['account']['simplesamlphp_sp_temp_bypass'] ?? NULL;
      $parents = is_array($element) && isset($element['#parents']) ? $element['#parents'] : NULL;
      $minute = $parents ? $form_state->getValue($parents) : NULL;
    }

    if (empty($minute)) {
      $minute = 0;
    }

    if (!is_numeric($minute)) {
      return;
    }

    $this->accountHelper->setTemporaryBypassMinutes($entity, (int) $minute);
  }

  /**
   * Validation handler preventing native login for SAML-managed accounts.
   */
  public function validateNativeLoginRestriction(array &$form, FormStateInterface $form_state): void {
    if (!$this->accountHelper->shouldRestrictNativeLogin()) {
      return;
    }

    $uid = $form_state->get('uid');
    if (empty($uid)) {
      return;
    }

    $debug_logging_enabled = $this->isDebugLoggingEnabled();

    $account = $this->accountHelper->loadAccountById((int) $uid);
    if (!$account instanceof UserInterface) {
      if ($debug_logging_enabled) {
        $this->logger->debug('Native login validation: unable to load account for uid @uid.', ['@uid' => (int) $uid]);
      }
      return;
    }

    $isSamlAccount = $this->accountHelper->isSamlAccount($account);
    $isExempt = $this->accountHelper->isExempt($account);

    if ($debug_logging_enabled) {
      $this->logger->debug('Native login validation for uid @uid: SAML managed = @managed, exempt = @exempt.', [
        '@uid' => $account->id(),
        '@managed' => $isSamlAccount ? 'yes' : 'no',
        '@exempt' => $isExempt ? 'yes' : 'no',
      ]);
    }

    if (!$isSamlAccount || $isExempt) {
      return;
    }

    $form_state->set('uid', FALSE);
    $form_state->setErrorByName('name', $this->t('Unrecognized username or password. <a href=":password">Forgot your password?</a>', [
      ':password' => Url::fromRoute('user.pass')->toString(),
    ]));
    $this->logger->notice('Blocked Drupal native login attempt for SAML provisioned account @uid.', ['@uid' => $account->id()]);
  }

  /**
   * Alters password reset confirmation forms for SAML-managed accounts.
   */
  protected function alterPasswordResetForm(array &$form, FormStateInterface $form_state): void {
    $account = $this->extractPasswordResetAccount($form_state);
    if (!$account instanceof UserInterface) {
      return;
    }

    if (!$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $form['simplesamlphp_sp_native_login_block'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['messages', 'messages--error']],
      '#weight' => -10,
      'message' => ['#markup' => $this->t('Password reset links cannot be used with accounts managed by Single Sign-On.')],
    ];

    $this->setElementAccess($form, ['actions', 'submit'], FALSE);
  }

  /**
   * Extracts the account associated with a password reset form build.
   */
  protected function extractPasswordResetAccount(FormStateInterface $form_state): ?UserInterface {
    $build_info = $form_state->getBuildInfo();
    $args = $build_info['args'] ?? [];
    $account = $args[0] ?? NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Sets the #access property on a nested form element when it exists.
   */
  protected function setElementAccess(array &$form, array $path, bool $value): void {
    $element = &$form;
    foreach ($path as $segment) {
      if (!isset($element[$segment])) {
        unset($element);
        return;
      }
      $element = &$element[$segment];
    }
    $element['#access'] = $value;
    unset($element);
  }

  /**
   * Indicates whether debug logging is enabled for the module.
   */
  protected function isDebugLoggingEnabled(): bool {
    return (bool) $this->configFactory
      ->get('simplesamlphp_sp.settings')
      ->get('log_debug_information');
  }

}
