<?php

namespace Drupal\simplesamlphp_sp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\externalauth\AuthmapInterface;
use Drupal\user\UserInterface;
use Drupal\user\UserDataInterface;

/**
 * Helper for identifying accounts managed by SimpleSAMLphp.
 */
class SimpleSamlAccountHelper {

  /**
   * User data key used to store temporary SAML bypass expiration timestamps.
   */
  private const TEMPORARY_BYPASS_KEY = 'temporary_bypass_expires';

  public function __construct(
    protected AuthmapInterface $authmap,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $configFactory,
    protected UserDataInterface $userData,
    protected TimeInterface $time,
  ) {}

  /**
   * Loads a user account by identifier.
   */
  public function loadAccountById(int $uid): ?UserInterface {
    $account = $this->entityTypeManager->getStorage('user')->load($uid);
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Loads a user account by username.
   */
  public function loadAccountByName(string $name): ?UserInterface {
    $accounts = $this->entityTypeManager->getStorage('user')->loadByProperties(['name' => $name]);
    $account = reset($accounts) ?: NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Loads a user account by email address.
   */
  public function loadAccountByMail(string $mail): ?UserInterface {
    $accounts = $this->entityTypeManager->getStorage('user')->loadByProperties(['mail' => $mail]);
    $account = reset($accounts) ?: NULL;
    return $account instanceof UserInterface ? $account : NULL;
  }

  /**
   * Determines whether the provided account is maintained by SimpleSAMLphp.
   */
  public function isSamlAccount(UserInterface $account): bool {
    if ($account->isNew()) {
      return FALSE;
    }

    return (bool) $this->authmap->getAuthData((int) $account->id(), SimpleSamlSpAuth::PROVIDER_ID);
  }

  /**
   * Determines whether the provided account is exempt from restrictions.
   */
  public function isExempt(UserInterface $account): bool {
    return in_array((int) $account->id(), $this->getExemptUserIds(), TRUE);
  }

  /**
   * Determines whether the provided account has an active SAML bypass.
   */
  public function hasActiveTemporaryBypass(UserInterface $account): bool {
    if ($account->isNew()) {
      return FALSE;
    }

    return $this->getRemainingTemporaryBypassMinutes($account) > 0;
  }

  /**
   * Returns remaining bypass minutes for the provided account.
   */
  public function getRemainingTemporaryBypassMinutes(UserInterface $account): int {
    if ($account->isNew()) {
      return 0;
    }

    $expires = $this->getTemporaryBypassExpiration($account);
    if ($expires <= 0) {
      return 0;
    }

    $now = $this->time->getRequestTime();
    if ($expires <= $now) {
      $this->clearTemporaryBypass($account);
      return 0;
    }

    $remaining_seconds = $expires - $now;
    return (int) ceil($remaining_seconds / 60);
  }

  /**
   * Sets a temporary SAML bypass for the given account.
   */
  public function setTemporaryBypassMinutes(UserInterface $account, int $minutes): void {
    if ($account->isNew()) {
      return;
    }

    $minutes = max(0, min(1440, $minutes));
    if ($minutes === 0) {
      $this->clearTemporaryBypass($account);
      return;
    }

    $expires = $this->time->getRequestTime() + ($minutes * 60);
    $this->userData->set('simplesamlphp_sp', (int) $account->id(), self::TEMPORARY_BYPASS_KEY, $expires);
  }

  /**
   * Clears any active temporary SAML bypass for the given account.
   */
  public function clearTemporaryBypass(UserInterface $account): void {
    if ($account->isNew()) {
      return;
    }

    $this->userData->delete('simplesamlphp_sp', (int) $account->id(), self::TEMPORARY_BYPASS_KEY);
  }

  /**
   * Retrieves the stored bypass expiration timestamp, if any.
   */
  public function getTemporaryBypassExpiration(UserInterface $account): int {
    if ($account->isNew()) {
      return 0;
    }

    $raw = $this->userData->get('simplesamlphp_sp', (int) $account->id(), self::TEMPORARY_BYPASS_KEY);
    return is_numeric($raw) ? (int) $raw : 0;
  }

  /**
   * Determines whether the provided account identifier is exempt.
   */
  public function isExemptById(int $uid): bool {
    return in_array($uid, $this->getExemptUserIds(), TRUE);
  }

  /**
   * Determines whether the provided account identifier is managed by SAML.
   */
  public function isSamlAccountById(int $uid): bool {
    $account = $this->loadAccountById($uid);
    return $account ? $this->isSamlAccount($account) : FALSE;
  }

  /**
   * Determines whether native Drupal login should be restricted for SAML users.
   */
  public function shouldRestrictNativeLogin(): bool {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    return !(bool) $config->get('allow_native_login_for_external_users');
  }

  /**
   * Retrieves the configured exempt user IDs.
   */
  public function getExemptUserIds(): array {
    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    $ids = $config->get('exempt_user_ids');
    if (!is_array($ids)) {
      $ids = [];
    }
    $ids[] = 1;
    $ids = array_unique(array_filter(array_map('intval', $ids), static fn($value) => $value > 0));
    sort($ids, SORT_NUMERIC);
    return array_values($ids);
  }

}
