<?php

namespace Drupal\simplesamlphp_sp\Controller;

use Drupal\Component\Utility\Unicode;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\PageCache\ResponsePolicy\KillSwitch;
use Drupal\Core\Render\Element\Email;
use Drupal\Core\Routing\TrustedRedirectResponse;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\simplesamlphp_sp\Exception\SamlLoginDeniedException;
use Drupal\simplesamlphp_sp\Service\SimpleSamlManager;
use Drupal\simplesamlphp_sp\Service\SimpleSamlSpAuth;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Handles the SAML login entry point and denial flow.
 */
class SamlController extends ControllerBase {

  /**
   * Module configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Service wrapper for SimpleSAMLphp interactions.
   *
   * @var \Drupal\simplesamlphp_sp\Service\SimpleSamlManager
   */
  protected $samlManager;

  /**
   * Current user account.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * Authentication helper responsible for linking/login.
   *
   * @var \Drupal\simplesamlphp_sp\Service\SimpleSamlSpAuth
   */
  protected $auth;

  /**
   * Page cache kill switch for disabling caching on SAML routes.
   *
   * @var \Drupal\Core\PageCache\ResponsePolicy\KillSwitch
   */
  protected $pageCacheKillSwitch;

  /**
   * Logger channel for SAML messages.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * SamlController constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Provides access to module configuration.
   * @param \Drupal\simplesamlphp_sp\Service\SimpleSamlManager $saml_manager
   *   Manages SimpleSAMLphp interactions.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   Current user session.
   * @param \Drupal\simplesamlphp_sp\Service\SimpleSamlSpAuth $auth
   *   Service for logging in/registering users via SAML.
   * @param \Drupal\Core\PageCache\ResponsePolicy\KillSwitch $pageCacheKillSwitch
   *   Kill switch for page caching.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   Logger for writing audit entries.
   */
  public function __construct(ConfigFactoryInterface $config_factory, SimpleSamlManager $saml_manager, AccountInterface $account, SimpleSamlSpAuth $auth, KillSwitch $pageCacheKillSwitch, LoggerChannelInterface $logger) {
    $this->config = $config_factory->get('simplesamlphp_sp.settings');
    $this->samlManager = $saml_manager;
    $this->account = $account;
    $this->auth = $auth;
    $this->pageCacheKillSwitch = $pageCacheKillSwitch;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('simplesamlphp_sp.manager'),
      $container->get('current_user'),
      $container->get('simplesamlphp_sp.auth'),
      $container->get('page_cache_kill_switch'),
      $container->get('logger.channel.simplesamlphp_sp')
    );
  }

  /**
   * Entry point for the SAML login path.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Redirect response to continue the authentication flow.
   */
  public function handleLogin() {
    // Disable page caching for this response.
    $this->pageCacheKillSwitch->trigger();

    if (!$this->samlManager->isActivated()) {
      return $this->redirect('user.login');
    }

    if ($this->account->isAuthenticated()) {
      return $this->redirect('user.page');
    }

    try {
      $attributes = $this->samlManager->requireAuth();

      if (empty($attributes)) {
        throw new AccessDeniedHttpException('No attributes received from the Identity Provider during SAML login.');
      }

      $unique_id_attr = $this->config->get('unique_id_attribute');
      $username_attr = $this->config->get('username_attribute');
      $email_attr = $this->config->get('email_attribute');

      $authname = !empty($attributes[$unique_id_attr][0]) ? $attributes[$unique_id_attr][0] : NULL;
      $username = !empty($attributes[$username_attr][0]) ? $attributes[$username_attr][0] : NULL;
      $email = !empty($attributes[$email_attr][0]) ? $attributes[$email_attr][0] : NULL;

      if (!$authname) {
        $this->logger->error('SAML login response did not contain the unique ID attribute (@attr).', ['@attr' => $unique_id_attr]);
        return $this->redirect('<front>');
      }

      if (!$email) {
        $this->logger->error('SAML login response did not contain the email attribute (@attr).', ['@attr' => $email_attr]);
        return $this->redirect('<front>');
      }

      // If no username is provided, fall back to the authname.
      $username = $username ?: $authname;

      $authname = $this->truncateAttributeValue($authname, UserInterface::USERNAME_MAX_LENGTH, 'authname');
      $username = $this->truncateAttributeValue($username, UserInterface::USERNAME_MAX_LENGTH, 'username');
      $email = $this->truncateAttributeValue($email, Email::EMAIL_MAX_LENGTH, 'email');

      $account = $this->auth->loginRegister($authname, $this->cleanUsername($username), $email);
      if ($account instanceof UserInterface) {
        $this->invokePostLoginHooks($account, $attributes, $authname, $username, $email);
        $this->logger->info('SAML login successful for user @authname', ['@authname' => $authname]);
      }
      return $this->redirect('user.page');
    }
    catch (SamlLoginDeniedException $e) {
      $this->logger->warning($e->getMessage());
      return $this->handleLoginDenial($e);
    }
    catch (\Exception $e) {
      $this->logger->error('SAML login failed: @m', ['@m' => $e->getMessage()]);
      return $this->redirect('<front>');
    }
  }

  /**
   * Invokes post-login hooks to allow user attribute synchronization.
   *
   * @param \Drupal\user\UserInterface $account
   *   The authenticated user account.
   * @param array $attributes
   *   IdP attributes returned during authentication.
   * @param string $authname
   *   Unique identifier from the SAML response.
   * @param string $username
   *   Username derived from the SAML response.
   * @param string $email
   *   Email address asserted by the IdP.
   */
  protected function invokePostLoginHooks(UserInterface $account, array $attributes, string $authname, string $username, string $email): void {
    $module_handler = $this->moduleHandler();
    if (!$module_handler) {
      return;
    }

    try {
      $module_handler->invokeAll('simplesamlphp_sp_post_login', [
        $account,
        $attributes,
        $authname,
        $username,
        $email,
      ]);
    }
    catch (\Throwable $exception) {
      $this->logger->error('A module threw an exception during SAML post-login processing: @message', [
        '@message' => $exception->getMessage(),
      ]);
    }
  }

  /**
   * Presents a denial message after IdP logout.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request containing denial context.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   */
  public function loginDenied(Request $request) {
    $reason = (string) $request->query->get('reason', 'generic');

    switch ($reason) {
      case 'user_1':
        $message = $this->t('The super administrator account cannot log in via SAML.');
        break;

      case 'restricted_role':
        $message = $this->t('Your account is not permitted to log in via SAML.');
        break;

      default:
        $message = $this->t('SAML login was denied for this account.');
        break;
    }

    throw new AccessDeniedHttpException($message);
  }

  /**
   * Initiates an IdP logout for denied accounts.
   *
   * @param \Drupal\simplesamlphp_sp\Exception\SamlLoginDeniedException $exception
   *   The raised denial exception.
   *
   * @return \Drupal\Core\Routing\TrustedRedirectResponse
   *   Redirect response to the denial page.
   */
  protected function handleLoginDenial(SamlLoginDeniedException $exception) {
    $return_url = Url::fromRoute('simplesamlphp_sp.login_denied', [], [
      'absolute' => TRUE,
      'query' => ['reason' => $exception->getReason()],
    ])->toString();

    $this->samlManager->logout($return_url);

    // In case logout does not redirect (e.g., no active IdP session),
    // fall back to an explicit redirect to the denial page.
    return new TrustedRedirectResponse($return_url);
  }

  /**
   * Truncates SAML attributes to the supported Drupal length.
   */
  protected function truncateAttributeValue(string $value, int $maxLength, string $attribute): string {
    $length = mb_strlen($value);

    if ($length > $maxLength) {
      $this->logger->warning('SAML @attribute value length (@length) exceeded the maximum @max characters and was truncated.', [
        '@attribute' => $attribute,
        '@length' => $length,
        '@max' => $maxLength,
      ]);

      return Unicode::truncate($value, $maxLength, FALSE, FALSE);
    }

    return $value;
  }

  /**
   * Normalizes the inbound username before registration.
   *
   * @param string $name
   *   Username provided by the IdP.
   *
   * @return string
   *   Sanitized username suitable for Drupal.
   */
  protected function cleanUsername($name) {
    $name = (string) preg_replace('/[^A-Za-z0-9_.-]/', '-', $name);

    return Unicode::truncate($name, UserInterface::USERNAME_MAX_LENGTH, FALSE, FALSE);
  }

}
