<?php

namespace Drupal\simplesamlphp_sp\EventSubscriber;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Blocks Drupal native login routes for SAML-managed accounts when configured.
 */
class SimpleSamlNativeLoginSubscriber implements EventSubscriberInterface {
  use StringTranslationTrait;

  public function __construct(
    protected SimpleSamlAccountHelper $accountHelper,
    protected MessengerInterface $messenger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['onKernelRequest', 32],
    ];
  }

  /**
   * Intercepts one-time login requests for SAML managed accounts.
   */
  public function onKernelRequest(RequestEvent $event): void {
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();
    $route = $request->attributes->get('_route');

    if (!$route || !$this->accountHelper->shouldRestrictNativeLogin()) {
      return;
    }

    switch ($route) {
      case 'user.reset.login':
        $this->interceptOneTimeLogin($event);
        break;

      case 'user.login.http':
        $this->interceptJsonLogin($event);
        break;

      case 'user.pass.http':
        $this->interceptJsonPasswordReset($event);
        break;
    }
  }

  /**
   * Prevents access to the one-time login route for SAML-managed accounts.
   */
  protected function interceptOneTimeLogin(RequestEvent $event): void {
    $request = $event->getRequest();
    $uid = (int) $request->attributes->get('uid');
    if ($uid <= 0) {
      return;
    }

    $account = $this->accountHelper->loadAccountById($uid);
    if (!$account || !$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    $this->messenger->addError($this->t('Password reset links cannot be used with accounts managed by Single Sign-On.'));
    $event->setResponse(new RedirectResponse(Url::fromRoute('user.login')->toString()));
  }

  /**
   * Prevents JSON login attempts for SAML-managed accounts.
   */
  protected function interceptJsonLogin(RequestEvent $event): void {
    $request = $event->getRequest();
    $payload = $this->decodeJsonPayload($request->getContent());
    $name = $payload['name'] ?? $payload['credentials']['name'] ?? NULL;

    if (!is_string($name) || $name === '') {
      return;
    }

    $account = $this->accountHelper->loadAccountByName($name);
    if (!$account || !$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    throw new BadRequestHttpException((string) $this->t('Sorry, unrecognized username or password.'));
  }

  /**
   * Prevents JSON password reset attempts for SAML-managed accounts.
   */
  protected function interceptJsonPasswordReset(RequestEvent $event): void {
    $request = $event->getRequest();
    $payload = $this->decodeJsonPayload($request->getContent());

    $account = NULL;
    if (!empty($payload['name']) && is_string($payload['name'])) {
      $account = $this->accountHelper->loadAccountByName($payload['name']);
    }
    elseif (!empty($payload['mail']) && is_string($payload['mail'])) {
      $account = $this->accountHelper->loadAccountByMail($payload['mail']);
    }

    if (!$account || !$this->accountHelper->isSamlAccount($account) || $this->accountHelper->isExempt($account)) {
      return;
    }

    throw new BadRequestHttpException('Unable to send email. Contact the site administrator if the problem persists.');
  }

  /**
   * Safely decodes a JSON request body.
   */
  protected function decodeJsonPayload(string $content): array {
    if ($content === '') {
      return [];
    }

    $decoded = json_decode($content, TRUE);
    return is_array($decoded) ? $decoded : [];
  }

}
