<?php

namespace Drupal\simplesamlphp_sp\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\externalauth\ExternalAuthInterface;
use Drupal\simplesamlphp_sp\Exception\SamlLoginDeniedException;
use Drupal\user\UserInterface;

/**
 * Service to link SimpleSAMLphp authentication with Drupal users.
 */
class SimpleSamlSpAuth {

  public const AUTHMAP_CREATED_FLAG = 'created_by_simplesamlphp_sp';

  public const PROVIDER_ID = 'simplesamlphp_sp';

  /**
   * The externalauth service.
   *
   * @var \Drupal\externalauth\ExternalAuthInterface
   */
  protected $externalAuth;

  /**
   * The user storage.
   *
   * @var \Drupal\user\UserStorageInterface
   */
  protected $userStorage;

  /**
   * Configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Helper service for inspecting SAML-managed accounts.
   */
  protected SimpleSamlAccountHelper $accountHelper;

  /**
   * Constructs a new SimpleSamlSpAuth object.
   *
   * @param \Drupal\externalauth\ExternalAuthInterface $external_auth
   *   The external auth service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\simplesamlphp_sp\Service\SimpleSamlAccountHelper $account_helper
   *   Helper service for SAML account checks.
   */
  public function __construct(ExternalAuthInterface $external_auth, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, SimpleSamlAccountHelper $account_helper) {
    $this->externalAuth = $external_auth;
    $this->userStorage = $entity_type_manager->getStorage('user');
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('simplesamlphp_sp');
    $this->accountHelper = $account_helper;
  }

  /**
   * Logs in a user, creating the user if it doesn't exist.
   *
   * @param string $authname
   *   The unique identifier from the SAML provider.
   * @param string $username
   *   The desired username for the user.
   * @param string $email
   *   The user's email address.
   *
   * @return \Drupal\user\UserInterface|null
   *   The logged in user object, or NULL on failure.
   *
   * @throws \Drupal\simplesamlphp_sp\Exception\SamlLoginDeniedException
   *   Thrown when the account is not permitted to log in via SAML.
   */
  public function loginRegister(string $authname, string $username, string $email): ?UserInterface {
    $account = $this->externalAuth->load($authname, static::PROVIDER_ID);

    if ($account) {
      $this->assertLoginAllowed($account);
      // Login the existing user.
      return $this->externalAuth->userLoginFinalize($account, $authname, static::PROVIDER_ID);
    }

    // Look for an existing user with the same email to link accounts.
    $existing_users = $this->userStorage->loadByProperties(['mail' => $email]);
    if ($existing_user = reset($existing_users)) {
      $this->assertLoginAllowed($existing_user);
      $this->externalAuth->linkExistingAccount($authname, static::PROVIDER_ID, $existing_user);
      return $this->externalAuth->userLoginFinalize($existing_user, $authname, static::PROVIDER_ID);
    }

    // Register a new user.
    try {
      $authmap_metadata = [static::AUTHMAP_CREATED_FLAG => TRUE];
      $account = $this->externalAuth->register($authname, static::PROVIDER_ID, ['mail' => $email, 'name' => $username], $authmap_metadata);
      if ($account) {
        $this->assertLoginAllowed($account);
        return $this->externalAuth->userLoginFinalize($account, $authname, static::PROVIDER_ID);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error registering user: @message', ['@message' => $e->getMessage()]);
    }

    return NULL;
  }

  /**
   * Validates whether the provided account may complete a SAML login.
   *
   * Denies access to the super administrator (user 1) and to any account that
   * holds a role listed in the `simplesamlphp_sp.settings:blocked_roles`
   * configuration (defaulting to `administrator` when undefined). Accounts with
   * an active temporary bypass skip the blocked-role constraint.
   *
   * @param \Drupal\user\UserInterface $account
   *   The account attempting to authenticate via SAML.
   *
   * @throws \Drupal\simplesamlphp_sp\Exception\SamlLoginDeniedException
   *   Thrown when the account is blocked from logging in via SAML.
   */
  protected function assertLoginAllowed(UserInterface $account): void {
    if ((int) $account->id() === 1) {
      $this->logger->warning('SAML login denied for user 1.');
      throw new SamlLoginDeniedException('user_1', 'The super administrator cannot log in via SAML.');
    }

    if ($this->accountHelper->hasActiveTemporaryBypass($account)) {
      return;
    }

    $config = $this->configFactory->get('simplesamlphp_sp.settings');
    $restricted_roles = $config->get('blocked_roles');
    if ($restricted_roles === NULL) {
      $restricted_roles = ['administrator'];
    }
    elseif (!is_array($restricted_roles)) {
      $restricted_roles = (array) $restricted_roles;
    }

    $restricted_roles = array_values(array_filter($restricted_roles));

    $matched_roles = $restricted_roles ? array_intersect($account->getRoles(), $restricted_roles) : [];
    if ($matched_roles) {
      $roles_list = implode(', ', $matched_roles);
      $message = 'SAML login denied for user (id:' . $account->id() . ') due to restricted role(s): ' . $roles_list;
      throw new SamlLoginDeniedException('restricted_role', $message);
    }
  }

}
