<?php

declare(strict_types=1);

namespace Drupal\site_assistant;

use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\RevisionLogInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Provides an interface defining an assistant entity.
 *
 * @ingroup site_assistant
 */
interface AssistantInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface, EntityPublishedInterface, RevisionLogInterface {

  /**
   * Gets the weight of this assistant.
   *
   * @return int
   *   The weight value.
   */
  public function getWeight(): int;

  /**
   * Sets the weight of this assistant.
   *
   * @param int $weight
   *   The weight value.
   *
   * @return $this
   */
  public function setWeight(int $weight): static;

  /**
   * Get the assistant content.
   *
   * @return \Drupal\Core\Field\FieldItemListInterface
   *   List of referenced entities.
   */
  public function getContent();

  /**
   * Gets the visibility conditions configuration.
   *
   * @return array<string, mixed>
   *   Array of visibility condition configurations.
   */
  public function getVisibility(): array;

  /**
   * Sets visibility condition configuration.
   *
   * @param string $instance_id
   *   The condition instance ID.
   * @param array<string, mixed> $configuration
   *   The condition configuration.
   *
   * @return $this
   */
  public function setVisibilityConfig(string $instance_id, array $configuration): static;

  /**
   * Gets the visibility conditions plugin collection.
   *
   * @return \Drupal\Core\Condition\ConditionPluginCollection
   *   The condition plugin collection.
   */
  public function &getVisibilityConditions(): ConditionPluginCollection;

  /**
   * Gets a specific visibility condition.
   *
   * @param string $instance_id
   *   The condition instance ID.
   *
   * @return \Drupal\Core\Condition\ConditionInterface
   *   The condition plugin instance.
   */
  public function getVisibilityCondition(string $instance_id): ConditionInterface;

  /**
   * Gets all visibility condition plugins.
   *
   * @return iterable<\Drupal\Core\Condition\ConditionInterface>
   *   An iterable of condition plugin instances.
   */
  public function getVisibilityPlugins(): iterable;

}
