<?php

namespace Drupal\site_assistant;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\site_assistant\DependencyInjection\AutowireEntityHandlerTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Defines a class to build a listing of assistant entities.
 *
 * @see \Drupal\site_assistant\Entity\AssistantLibraryItem
 */
class AssistantLibraryItemListBuilder extends EntityListBuilder {

  use AutowireEntityHandlerTrait;

  /**
   * Constructs a new AssistantLibraryItemListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    #[Autowire('entity_type.manager')]
    protected EntityTypeManagerInterface $entityTypeManager,
    #[Autowire('date.formatter')]
    protected DateFormatterInterface $dateFormatter,
    #[Autowire('language_manager')]
    protected LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($entity_type, $storage);
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    // Enable language column and filter if multiple languages are added.
    $header = [
      'title' => $this->t('Title'),
      'author' => [
        'data' => $this->t('Author'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'status' => $this->t('Status'),
      'changed' => [
        'data' => $this->t('Updated'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
    ];
    if ($this->languageManager->isMultilingual()) {
      $header['language_name'] = [
        'data' => $this->t('Language'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ];
    }
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\site_assistant\Entity\AssistantLibraryItem $entity */
    $langcode = $entity->language()->getId();
    $languages = $this->languageManager->getLanguages();

    $uri = $entity->toUrl();
    $options = $uri->getOptions();
    $options += ($langcode !== LanguageInterface::LANGCODE_NOT_SPECIFIED && isset($languages[$langcode]) ? ['language' => $languages[$langcode]] : []);
    $uri->setOptions($options);

    /**
     * @var array{
     *  title: array<string, array>,
     *  author: array<string, array>,
     *  status: \Drupal\Core\StringTranslation\TranslatableMarkup,
     *  changed: \Drupal\Core\StringTranslation\TranslatableMarkup,
     *  language_name?: array<string, string>,
     *  operations: array<string, array>,
     *  ...
     * } $row
     */
    $row = [];
    $row['title']['data'] = [
      '#type' => 'link',
      '#title' => $entity->label(),
      '#url' => $uri,
    ];
    $row['author']['data'] = [
      '#theme' => 'username',
      '#account' => $entity->getOwner(),
    ];
    $row['status'] = $entity->isPublished() ? $this->t('published') : $this->t('not published');

    $changed = $entity->getChangedTime();
    $row['changed'] = is_null($changed) ? $this->t('n/a') : $this->dateFormatter->format($changed, 'short');

    if ($this->languageManager->isMultilingual()) {
      $row['language_name'] = $this->languageManager->getLanguageName($langcode);
    }

    $row['operations']['data'] = $this->buildOperations($entity);

    return $row + parent::buildRow($entity);
  }

}
