<?php

namespace Drupal\site_assistant\DependencyInjection;

use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\Exception\AutowiringFailedException;

/**
 * Defines a trait for automatically wiring dependencies from the container.
 *
 * This trait uses reflection and may cause performance issues with classes
 * that will be instantiated multiple times.
 */
trait AutowireEntityHandlerTrait {

  /**
   * Instantiates a new instance of the implementing class using autowiring.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container this instance should use.
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return static
   *   The current handler instance.
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): static {
    // @phpstan-ignore-next-line
    if (!method_exists(static::class, '__construct')) {
      // @phpstan-ignore-next-line
      return new static($container, $entity_type);
    }

    $storage = $container->get('entity_type.manager')->getStorage($entity_type->id());

    // Prepare constructor arguments.
    $args = [];
    $constructor = new \ReflectionMethod(static::class, '__construct');
    foreach ($constructor->getParameters() as $parameter) {
      $args[$parameter->getName()] = match ($parameter->getName()) {
        // Allow constructor to use either snake_case or camelCase.
        'entity_type', 'pluginType' => $entity_type,
        'storage' => $storage,
        default => (static function () use ($container, $parameter) {
          $service = ltrim((string) $parameter->getType(), '?');
          foreach ($parameter->getAttributes(Autowire::class) as $attribute) {
            // @phpstan-ignore-next-line
            $service = (string) $attribute->newInstance()->value;
          }

          if (!$container->has($service)) {
            throw new AutowiringFailedException($service, sprintf('Cannot autowire service "%s": argument "$%s" of method "%s::_construct()", you should configure its value explicitly.', $service, $parameter->getName(), static::class));
          }

          return $container->get($service);
        })(),
      };
    }

    // @phpstan-ignore-next-line
    return new static(...$args);
  }

}
