<?php

namespace Drupal\site_assistant\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\site_assistant\AssistantListEntryAccessControlHandler;
use Drupal\site_assistant\AssistantListEntryInterface;
use Drupal\site_assistant\Traits\EntityCreatedTrait;
use Drupal\site_assistant\Traits\SiteAssistantRevisionTrait;
use Drupal\site_assistant\Traits\SiteAssistantTranslationTrait;
use Drupal\user\Entity\EntityPermissionsRouteProvider;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the assistant_list_entry entity.
 *
 * @ingroup site_assistant
 */
#[ContentEntityType(
  id: 'assistant_list_entry',
  label: new TranslatableMarkup('Assistant List Entry'),
  handlers: [
    'form' => [
      'default' => ContentEntityForm::class,
      'add' => ContentEntityForm::class,
      'edit' => ContentEntityForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
      'permissions' => EntityPermissionsRouteProvider::class,
    ],
    'access' => AssistantListEntryAccessControlHandler::class,
    'views_data' => EntityViewsData::class,
  ],
  base_table: 'assistant_list_entry',
  data_table: 'assistant_list_entry_field_data',
  revision_table: 'assistant_list_entry_revision',
  revision_data_table: 'assistant_list_entry_field_revision',
  show_revision_ui: TRUE,
  translatable: TRUE,
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
    'bundle' => 'bundle',
    'revision' => 'vid',
    'label' => 'title',
    'langcode' => 'langcode',
    'status' => 'status',
    'published' => 'status',
    'uid' => 'uid',
    'owner' => 'uid',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_user',
    'revision_created' => 'revision_created',
    'revision_log_message' => 'revision_log_message',
  ],
  links: [
    'canonical' => '/assistant_list_entry/{assistant_list_entry]',
    'add-page' => '/assistant_list_entry/add',
    'add-form' => '/assistant_list_entry/add/{assistant_list_entry_type]',
    'edit-form' => '/assistant_list_entry/{assistant_list_entry]/edit',
    'delete-form' => '/assistant_list_entry/{assistant_list_entry]/delete',
    'collection' => '/admin/content/assistant_list_entries',
  ],
  admin_permission: 'administer site_assistant',
  bundle_entity_type: 'assistant_list_entry_type',
  field_ui_base_route: 'entity.assistant_list_entry_type.edit_form',
)]
#[\AllowDynamicProperties]
class AssistantListEntry extends EditorialContentEntityBase implements AssistantListEntryInterface {

  use EntityCreatedTrait;
  use EntityOwnerTrait;
  use SiteAssistantRevisionTrait;
  use SiteAssistantTranslationTrait;

  /**
   * {@inheritdoc}
   *
   * When a new entity instance is added, set the uid entity reference to
   * the current user as the creator of the instance.
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values): void {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    $this->enforceTranslationOwnerId();
    $this->setRevisionOwnerId();
  }

  /**
   * {@inheritdoc}
   *
   * Define the field properties here.
   *
   * Field name, type and size determine the table structure.
   *
   * In addition, we can define how the field and its content can be manipulated
   * in the GUI. The behavior of the widgets used can be determined here.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    // Standard field, used as unique if primary index.
    $fields['id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('ID'))
      ->setDescription(t('The ID of the Help Section entity.'))
      ->setReadOnly(TRUE);

    // Standard field, unique outside of the scope of the current project.
    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setDescription(t('The UUID of the Help Section entity.'))
      ->setReadOnly(TRUE);

    $fields['bundle'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel($entity_type->getBundleLabel())
      ->setSetting('target_type', $entity_type->getBundleEntityType())
      ->setRequired(TRUE)
      ->setReadOnly(TRUE);

    $fields['uid']
      ->setLabel(t('Authored by'))
      ->setDescription(t('The username of the content author.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'region' => 'hidden',
      ]);

    $fields['status']
      ->setDisplayOptions('form', [
        'region' => 'hidden',
      ]);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language code'))
      ->setDescription(t('The language code of assist list entry entity.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));
    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    // Standard field, used as unique if primary index.
    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Title'))
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'weight' => -99,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'text_textfield',
        'weight' => -99,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE);

    return $fields;
  }

}
