<?php

declare(strict_types=1);

namespace Drupal\site_assistant\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Plugin\ContextAwarePluginInterface;
use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\site_assistant\Traits\SiteAssistantFormAuthorTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormGeneralTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormMetaTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormOptionsTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Form controller for the site_assistant entity edit forms.
 *
 * @ingroup site_assistant
 */
class AssistantForm extends ContentEntityForm {

  use AutowireTrait;
  use SiteAssistantFormAuthorTrait;
  use SiteAssistantFormGeneralTrait;
  use SiteAssistantFormMetaTrait;
  use SiteAssistantFormOptionsTrait;

  /**
   * AssistantForm constructor.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   Entity repository.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language
   *   Language manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   Current user.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   Date formatter.
   * @param \Drupal\Core\Condition\ConditionManager $conditionManager
   *   The condition plugin manager.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $contextRepository
   *   The context repository.
   */
  public function __construct(
    #[Autowire('entity.repository')]
    EntityRepositoryInterface $entity_repository,
    #[Autowire('entity_type.bundle.info')]
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    #[Autowire('datetime.time')]
    TimeInterface $time,
    #[Autowire('language_manager')]
    protected LanguageManagerInterface $language,
    #[Autowire('current_user')]
    protected AccountInterface $currentUser,
    #[Autowire('date.formatter')]
    protected DateFormatterInterface $dateFormatter,
    #[Autowire('plugin.manager.condition')]
    protected ConditionManager $conditionManager,
    #[Autowire('context.repository')]
    protected ContextRepositoryInterface $contextRepository,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Store the gathered contexts in the form state for other objects to use
    // during form building.
    $form_state->setTemporaryValue('gathered_contexts', $this->contextRepository->getAvailableContexts());

    // Initialize conditions early - before parent::buildForm().
    $this->initializeConditions($form_state);

    $form = parent::buildForm($form, $form_state);
    /** @var \Drupal\site_assistant\AssistantInterface $entity */
    $entity = $this->entity;

    $this->addFormElementGeneral($entity, $form);
    $this->addFormElementMeta($entity, $form);
    $this->addFormElementAuthor($entity, $form);
    $this->addFormElementOptions($entity, $form);

    /** @var array{
     * '#attached'?: array<string, array>,
     * 'advanced'?: array<string, array<string, array>>,
     * 'revision_information'?: array,
     * 'status'?: array,
     * 'visibility'?: array<string, array>,
     * 'visibility_plugins'?: array,
     * ...
     * } $form */
    $form['advanced']['#attributes']['class'][] = 'entity-meta';
    $form['advanced']['#type'] = 'container';

    $form['status']['#group'] = 'footer';

    $form['#attached']['library'][] = 'site_assistant/form';

    $form['revision_information']['#access'] = FALSE;

    // Build visibility conditions section.
    $this->buildVisibilitySection($form, $form_state);

    return $form;
  }

  /**
   * Initializes the visibility conditions in form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function initializeConditions(FormStateInterface $form_state): void {
    // Only initialize once.
    if ($form_state->get('visibility_conditions') !== NULL) {
      return;
    }

    /** @var \Drupal\site_assistant\AssistantInterface $entity */
    $entity = $this->entity;
    $conditions = [];

    // Load existing conditions from entity.
    foreach ($entity->getVisibilityPlugins() as $instance_id => $plugin) {
      $conditions[$instance_id] = $plugin->getConfiguration();
    }

    $form_state->set('visibility_conditions', $conditions);
  }

  /**
   * Builds the visibility conditions section.
   *
   * @param mixed[] $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function buildVisibilitySection(array &$form, FormStateInterface $form_state): void {
    $wrapper_id = 'visibility-conditions-wrapper';

    // Get conditions from form state.
    $conditions = $form_state->get('visibility_conditions') ?? [];
    $has_conditions = (count($conditions) > 0);

    $form['visibility'] = [
      '#type' => 'details',
      '#open' => $has_conditions,
      '#title' => $this->t('Visibility conditions'),
      '#tree' => TRUE,
    ];

    // Container for existing conditions.
    $form['visibility']['conditions'] = [
      '#type' => 'container',
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    // Build form elements for each condition.
    foreach ($conditions as $instance_id => $condition_config) {
      $form['visibility']['conditions'][$instance_id] = $this->buildConditionElement(
        condition_config: $condition_config,
        instance_id: $instance_id,
        wrapper_id: $wrapper_id,
        form: $form,
        form_state: $form_state,
      );
    }

    // Get available condition plugins.
    $condition_definitions = $this->conditionManager->getDefinitions();
    // $used_plugin_ids = array_column($conditions, 'id');
    $condition_options = ['' => $this->t('- Select a condition -')];

    foreach ($condition_definitions as $plugin_id => $definition) {
      $condition_options[$plugin_id] = $definition['label'];
    }

    // Add new condition selector.
    $form['visibility']['add_condition'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['container-inline']],
    ];

    $form['visibility']['add_condition']['plugin_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Condition type'),
      '#options' => $condition_options,
      '#wrapper_id' => $wrapper_id,
      // Disable if all conditions are already added.
      '#disabled' => count($condition_options) <= 1,
    ];

    $form['visibility']['add_condition']['add_button'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add condition'),
      '#submit' => [[$this, 'addConditionSubmit']],
      '#ajax' => [
        'callback' => '::ajaxAddCondition',
        'wrapper' => $wrapper_id,
      ],
      '#limit_validation_errors' => [],
      '#disabled' => count($condition_options) <= 1,
    ];

    if (count($condition_options) <= 1) {
      $form['visibility']['add_condition']['plugin_id']['#description'] = $this->t('All available conditions have been added.');
    }
  }

  /**
   * Submit handler to add a condition.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function addConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $user_input = $form_state->getUserInput();

    $plugin_id = $user_input['visibility']['add_condition']['plugin_id'] ?? NULL;

    if (isset($plugin_id) && is_string($plugin_id)) {
      $conditions = $form_state->get('visibility_conditions') ?? [];

      // Only add if not already present.
      $existing_plugin_ids = array_column($conditions, 'id');
      if (!in_array($plugin_id, $existing_plugin_ids, TRUE)) {
        // Use plugin_id as instance_id for uniqueness.
        $conditions[$plugin_id] = [
          'id' => $plugin_id,
        ];
        $form_state->set('visibility_conditions', $conditions);

        $form_state->setRebuild(TRUE);
      }

      // Clear the selector value in the form state values array.
      $form_state->setValue(['visibility', 'add_condition', 'plugin_id'], '');
    }
  }

  /**
   * AJAX callback to add a new condition.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form element to return.
   */
  public function ajaxAddCondition(array &$form, FormStateInterface $form_state): array {
    // Rely on addConditionSubmit to call setRebuild(TRUE) and rebuild the form.
    return $form['visibility']['conditions'];
  }

  /**
   * AJAX callback to remove a condition.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form element to return.
   */
  public function ajaxRemoveCondition(array &$form, FormStateInterface $form_state): array {
    // We rely on removeConditionSubmit to call setRebuild(TRUE) and rebuild
    // the form.
    return $form['visibility']['conditions'];
  }

  /**
   * Builds a form element for a single condition.
   *
   * @param array<string, mixed> $condition_config
   *   The condition configuration.
   * @param string $instance_id
   *   The condition instance ID.
   * @param string $wrapper_id
   *   The AJAX wrapper ID.
   * @param mixed[] $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The condition form element.
   */
  protected function buildConditionElement(array $condition_config, string $instance_id, string $wrapper_id, array &$form, FormStateInterface $form_state): array {
    $plugin_id = $condition_config['id'] ?? '';

    if (!is_string($plugin_id) || (strlen($plugin_id) === 0)) {
      return [];
    }

    $plugin_definition = $this->conditionManager->getDefinition($plugin_id, FALSE);
    $label = $plugin_definition['label'] ?? $plugin_id;

    // Use a details element to wrap the condition form.
    $element = [
      '#type' => 'details',
      '#title' => $label,
      '#open' => TRUE,
      '#attributes' => ['class' => ['visibility-condition-item']],
    ];

    // Hidden field to store plugin ID.
    $element['id'] = [
      '#type' => 'hidden',
      '#value' => $plugin_id,
    ];

    // Create the condition plugin instance.
    try {
      /** @var \Drupal\Core\Condition\ConditionInterface $condition */
      $condition = $this->conditionManager->createInstance($plugin_id, $condition_config);
      $this->setConditionContext($condition);

      // Build the condition configuration form.
      $subform_state = SubformState::createForSubform($element, $form, $form_state);
      $condition_form = $condition->buildConfigurationForm([], $subform_state);

      // Only add configuration if there are form elements.
      if (count($condition_form) > 0) {
        $element['configuration'] = $condition_form;
        $element['configuration']['#tree'] = TRUE;
      }
    }
    catch (\Exception $e) {
      $this->logger('site_assistant')->error('Error creating condition plugin @plugin_id: @message', [
        '@plugin_id' => $plugin_id,
        '@message' => $e->getMessage(),
      ]);
      $element['error'] = [
        '#markup' => $this->t('Error loading condition plugin: @error', ['@error' => $e->getMessage()]),
      ];
    }

    // Remove button.
    $element['remove'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove'),
      '#name' => 'remove_condition_' . $plugin_id,
      '#submit' => [[$this, 'removeConditionSubmit']],
      '#ajax' => [
        'callback' => '::ajaxRemoveCondition',
        'wrapper' => $wrapper_id,
      ],
      '#limit_validation_errors' => [],
      '#instance_id' => $plugin_id,
      '#wrapper_id' => $wrapper_id,
      '#weight' => 100,
    ];

    return $element;
  }

  /**
   * Set contexts required for a condition.
   *
   * @param \Drupal\Core\Condition\ConditionInterface $condition
   *   The condition plugin.
   */
  protected function setConditionContext(ConditionInterface $condition): void {
    if (!($condition instanceof ContextAwarePluginInterface)) {
      return;
    }

    // Inject context for conditions that need it.
    $context_definitions = $condition->getContextDefinitions();
    if (count($context_definitions) === 0) {
      return;
    }

    try {
      // Get available contexts from the context repository.
      $contexts = $this->contextRepository->getAvailableContexts();

      // Set contexts that match the condition's requirements.
      foreach ($context_definitions as $context_name => $context_definition) {
        $context_data_type = $context_definition->getDataType();

        // Find a matching context.
        foreach ($contexts as $context) {
          if ($context->getContextDefinition()->getDataType() === $context_data_type) {
            $condition->setContext($context_name, $context);
            break;
          }
        }
      }
    }
    catch (\Exception $e) {
      // Context not available, continue without it.
      $this->logger('site_assistant')->debug('Could not set context for condition @plugin_id: @message', [
        '@plugin_id' => $condition->getPluginId(),
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Submit handler to remove a condition.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function removeConditionSubmit(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $instance_id = $triggering_element['#instance_id'] ?? NULL;

    $conditions = $form_state->get('visibility_conditions') ?? [];

    // Remove the condition with the specified instance_id.
    if (isset($conditions[$instance_id])) {
      unset($conditions[$instance_id]);
      unset($form['visibility']['conditions'][$instance_id]);
      $form_state->set('visibility_conditions', $conditions);
      $form_state->setRebuild(TRUE);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    /** @var \Drupal\site_assistant\AssistantInterface $entity */
    $entity = & $this->entity;

    // Reset all plugins to allow removing conditions.
    $conditions = & $entity->getVisibilityConditions();
    foreach ($entity->getVisibilityPlugins() as $plugin_id => $plugin) {
      $conditions->removeInstanceId($plugin_id);
    }

    // Process visibility conditions and save to entity.
    $submitted_conditions = $form_state->getValue(['visibility', 'conditions']) ?? [];

    // Iterate over the submitted form values.
    foreach ($submitted_conditions as $condition_data) {
      // Ensure the condition data is valid and has an ID.
      if (!is_array($condition_data) || !isset($condition_data['id'])) {
        continue;
      }

      $plugin_id = $condition_data['id'];

      // Start with the plugin ID.
      $config = ['id' => $plugin_id];

      // Add configuration values submitted under the 'configuration' key.
      if (isset($condition_data['configuration']) && is_array($condition_data['configuration'])) {
        // Merge the configuration array directly. This ensures field values
        // are saved.
        $config += $condition_data['configuration'];
      }

      // Add negate value if present.
      if (isset($condition_data['negate'])) {
        $config['negate'] = (bool) $condition_data['negate'];
      }

      // Set the processed conditions on the entity.
      $entity->setVisibilityConfig($plugin_id, $config);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $status = parent::save($form, $form_state);

    $entity = $this->entity;
    if ($status === SAVED_UPDATED) {
      $this->messenger()->addMessage($this->t('The Assist %assistant has been updated.', ['%assistant' => $entity->toLink()->toString()]));
    }
    else {
      $this->messenger()->addMessage($this->t('The Assist %assistant has been added.', ['%assistant' => $entity->toLink()->toString()]));
    }

    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $status;
  }

}
