<?php

namespace Drupal\site_assistant\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\site_assistant\DependencyInjection\AutowirePluginTrait;
use Drupal\site_assistant\SiteAssistantInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Provides a block that shows the assist for this current page request.
 */
#[Block(
  id: 'site_assistant_assist_block',
  admin_label: new TranslatableMarkup('Site Assistant: Current assist')
)]
class SiteAssistantBlock extends BlockBase implements ContainerFactoryPluginInterface {

  use AutowirePluginTrait;

  /**
   * SiteAssistantBlock constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\site_assistant\SiteAssistantInterface $assistantService
   *   Assist service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    $plugin_definition,
    #[Autowire('site_assistant')]
    protected SiteAssistantInterface $assistantService,
    #[Autowire('entity_type.manager')]
    protected EntityTypeManagerInterface $entityTypeManager,
    #[Autowire('language_manager')]
    protected LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return ['label_display' => FALSE];
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $active_assistant = $this->assistantService->getActiveAssistant();

    if (is_null($active_assistant)) {
      return [];
    }

    $list_entries = [];

    $view_builder = $this->entityTypeManager->getViewBuilder('assistant_list_entry');

    // Get current language.
    $langcode = $this->languageManager->getCurrentLanguage()->getId();

    if ($active_assistant->hasTranslation($langcode)) {
      try {
        /** @var \Drupal\site_assistant\AssistantInterface $translated_active_assistant */
        $translated_active_assistant = $active_assistant->getTranslation($langcode);
        $active_assistant = $translated_active_assistant;
      }
      catch (\InvalidArgumentException $exc) {
        // Simply ignore and use the default translation.
      }
    }

    /** @var \Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem $assistant_list_entry_reference */
    foreach ($active_assistant->getContent() as $assistant_list_entry_reference) {
      if (!is_null($assistant_list_entry_reference->entity)) {
        $list_entries[] = $view_builder->view($assistant_list_entry_reference->entity);
      }
    }

    $renderable = [
      '#theme' => 'site_assistant',
      '#list_entries' => $list_entries,
    ];

    return $renderable;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $contexts = parent::getCacheContexts();
    $active_assistant = $this->assistantService->getActiveAssistant();
    if (is_null($active_assistant)) {
      return $contexts;
    }

    return Cache::mergeContexts($contexts, $active_assistant->getCacheContexts());
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = parent::getCacheTags();
    $active_assistant = $this->assistantService->getActiveAssistant();
    if (is_null($active_assistant)) {
      return $tags;
    }

    return Cache::mergeTags($tags, $active_assistant->getCacheTags());
  }

}
