<?php

namespace Drupal\site_assistant;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\site_assistant\DependencyInjection\AutowireEntityHandlerTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Defines a class to build a listing of assistant entities.
 *
 * @see \Drupal\site_assistant\Entity\Assistant
 */
class AssistantListBuilder extends EntityListBuilder implements FormInterface {

  use AutowireEntityHandlerTrait;

  /**
   * The key to use for the form element containing the entities.
   *
   * @var string
   */
  protected $entitiesKey = 'assistants';

  /**
   * The entities being listed.
   *
   * @var \Drupal\site_assistant\AssistantInterface[]
   */
  protected $entities = [];

  /**
   * Name of the entity's weight field or FALSE if no field is provided.
   *
   * @var string|bool
   */
  protected $weightKey = 'weight';

  /**
   * {@inheritdoc}
   */
  protected $limit = FALSE;

  /**
   * Constructs a new AssistantListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   The date formatter service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder service.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    #[Autowire('entity_type.manager')]
    protected EntityTypeManagerInterface $entityTypeManager,
    #[Autowire('date.formatter')]
    protected DateFormatterInterface $dateFormatter,
    #[Autowire('language_manager')]
    protected LanguageManagerInterface $languageManager,
    #[Autowire('form_builder')]
    protected FormBuilderInterface $formBuilder,
  ) {
    parent::__construct($entity_type, $storage);

    // Check if the entity type supports weighting.
    if ($this->entityType->hasKey('weight')) {
      $this->weightKey = $this->entityType->getKey('weight');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    // Enable language column and filter if multiple languages are added.
    $header = [
      'title' => $this->t('Title'),
      'author' => [
        'data' => $this->t('Author'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      'changed' => [
        'data' => $this->t('Updated'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
    ];
    if ($this->languageManager->isMultilingual()) {
      $header['language_name'] = [
        'data' => $this->t('Language'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ];
    }
    if ($this->hasWeightKeySet()) {
      $header['weight'] = $this->t('Weight');
    }
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\site_assistant\Entity\Assistant $entity */
    $langcode = $entity->language()->getId();
    $languages = $this->languageManager->getLanguages();

    $uri = $entity->toUrl();
    $options = $uri->getOptions();
    $options += ($langcode !== LanguageInterface::LANGCODE_NOT_SPECIFIED && isset($languages[$langcode]) ? ['language' => $languages[$langcode]] : []);
    $uri->setOptions($options);

    /**
     * @var array{
     *  title: array<string, array>,
     *  author: array<string, array>,
     *  changed: array<\Drupal\Core\StringTranslation\TranslatableMarkup>,
     *  language_name?: array<string, string>,
     *  ...
     * } $row
     */
    $row = [];
    $row['title']['data'] = [
      '#type' => 'link',
      '#title' => $entity->label(),
      '#url' => $uri,
    ];
    $row['author']['data'] = [
      '#theme' => 'username',
      '#account' => $entity->getOwner(),
    ];

    $changed = $entity->getChangedTime();
    $row['changed'] = [
      '#markup' => is_null($changed) ? $this->t('n/a') : $this->dateFormatter->format($changed, 'short'),
    ];

    if ($this->languageManager->isMultilingual()) {
      $row['language_name'] = [
        '#markup' => $this->languageManager->getLanguageName($langcode),
      ];
    }

    if ($this->hasWeightKeySet()) {
      // Override default values to markup elements.
      $row['#attributes']['class'][] = 'draggable';
      $row['#weight'] = $entity->get($this->weightKey)->value;
      // Add weight column.
      $row['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @title', ['@title' => $entity->label()]),
        '#title_display' => 'invisible',
        '#default_value' => $entity->get($this->weightKey)->value,
        '#attributes' => ['class' => ['assistant-section-weight']],
      ];
    }

    $row['operations']['data'] = $this->buildOperations($entity);

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    if ($this->hasWeightKeySet()) {
      return $this->formBuilder->getForm($this);
    }

    return parent::render();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'assistant_list_weight_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form[$this->entitiesKey] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#empty' => $this->t('There are no @label yet.', ['@label' => $this->entityType->getPluralLabel()]),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'assistant-section-weight',
        ],
      ],
    ];

    $this->entities = $this->load();
    $delta = 10;
    // Change the delta of the weight field if there are more than 20 entities.
    if ($this->hasWeightKeySet()) {
      $count = count($this->entities);
      if ($count > 20) {
        $delta = ceil($count / 2);
      }
    }
    foreach ($this->entities as $entity) {
      $row = $this->buildRow($entity);
      if (isset($row['label'])) {
        $row['label'] = ['#markup' => $row['label']];
      }
      if (isset($row['weight'])) {
        $row['weight']['#delta'] = $delta;
      }
      $form[$this->entitiesKey][$entity->id()] = $row;
    }

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    // No validation.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    foreach ($form_state->getValue($this->entitiesKey) as $id => $value) {
      if (isset($this->entities[$id]) && $this->entities[$id]->get($this->weightKey) !== $value['weight']) {

        /** @var \Drupal\site_assistant\AssistantInterface $assistant */
        $assistant = $this->entities[$id];

        $assistant->setWeight($value['weight']);
        $assistant->save();
      }
    }
  }

  /**
   * Loads entities of this type from storage for listing.
   *
   * @return \Drupal\site_assistant\AssistantInterface[]
   *   An array of entities implementing
   *   \Drupal\site_assistant\AssistantInterface indexed by their IDs.
   *   Returns an empty array if no matching entities are found.
   */
  public function load(): array {
    $entity_ids = $this->getEntityIds();
    /** @var \Drupal\site_assistant\AssistantInterface[] $entities */
    $entities = $this->storage->loadMultiple($entity_ids);

    uasort($entities, function ($a, $b) {
      /** @var \Drupal\site_assistant\AssistantInterface $a */
      /** @var \Drupal\site_assistant\AssistantInterface $b */
      if ($a->getWeight() === $b->getWeight()) {
        return 0;
      }
      return ($a->getWeight() < $b->getWeight()) ? -1 : 1;
    });

    return $entities;
  }

  /**
   * Check if a weight key is set.
   *
   * @return bool
   *   <code>TRUE</code> if a weight key has been set, otherwise
   *   <code>FALSE</code>.
   */
  protected function hasWeightKeySet(): bool {
    return (is_string($this->weightKey) && (strlen($this->weightKey) > 0));
  }

}
