<?php

declare(strict_types=1);

namespace Drupal\site_assistant\Entity;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Condition\ConditionInterface;
use Drupal\Core\Condition\ConditionPluginCollection;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityViewBuilder;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Plugin\ObjectWithPluginCollectionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\site_assistant\AssistantAccessControlHandler;
use Drupal\site_assistant\AssistantInterface;
use Drupal\site_assistant\AssistantListBuilder;
use Drupal\site_assistant\Form\AssistantDeleteForm;
use Drupal\site_assistant\Form\AssistantForm;
use Drupal\site_assistant\Traits\EntityCreatedTrait;
use Drupal\site_assistant\Traits\SiteAssistantRevisionTrait;
use Drupal\site_assistant\Traits\SiteAssistantTranslationTrait;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the assistant entity.
 *
 * @ingroup site_assistant
 */
#[ContentEntityType(
  id: 'assistant',
  label: new TranslatableMarkup('Assistant'),
  handlers: [
    'view_builder' => EntityViewBuilder::class,
    'list_builder' => AssistantListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'default' => AssistantForm::class,
      'edit' => AssistantForm::class,
      'delete' => AssistantDeleteForm::class,
    ],
    'access' => AssistantAccessControlHandler::class,
  ],
  admin_permission: 'administer site_assistant',
  base_table: 'assistant',
  data_table: 'assistant_field_data',
  revision_table: 'assistant_revision',
  revision_data_table: 'assistant_field_revision',
  show_revision_ui: TRUE,
  translatable: TRUE,
  entity_keys: [
    'id' => 'id',
    'revision' => 'vid',
    'label' => 'title',
    'langcode' => 'langcode',
    'uuid' => 'uuid',
    'status' => 'status',
    'published' => 'status',
    'uid' => 'uid',
    'owner' => 'uid',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_user',
    'revision_created' => 'revision_created',
    'revision_log_message' => 'revision_log_message',
  ],
  links: [
    'canonical' => '/site_assistant/assistant/{assistant]',
    'edit-form' => '/site_assistant/assistant/{assistant]/edit',
    'delete-form' => '/site_assistant/assistant/{assistant]/delete',
    'revision' => '/site_assistant/assistant/{assistant]/revisions/{assistant_revision]/view',
    'collection' => '/site_assistant/assistant/list',
  ],
)]
class Assistant extends EditorialContentEntityBase implements AssistantInterface, ObjectWithPluginCollectionInterface {

  use EntityCreatedTrait;
  use EntityOwnerTrait;
  use SiteAssistantRevisionTrait;
  use SiteAssistantTranslationTrait;

  /**
   * The visibility collection.
   *
   * @var \Drupal\Core\Condition\ConditionPluginCollection|null
   */
  protected ?ConditionPluginCollection $visibilityCollection = NULL;

  /**
   * The condition plugin manager.
   *
   * @var \Drupal\Core\Condition\ConditionManager|null
   */
  protected $conditionPluginManager = NULL;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values): void {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    $this->enforceTranslationOwnerId();
    $this->setRevisionOwnerId();

    // Sync visibility plugin collection back to field before saving.
    if (isset($this->visibilityCollection)) {
      $visibility_values = [];
      foreach ($this->visibilityCollection as $condition) {
        assert($condition instanceof ConditionInterface);
        $visibility_values[] = $condition->getConfiguration();
      }
      $this->set('visibility_plugins', $visibility_values);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    $fields['id'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('ID'))
      ->setDescription(new TranslatableMarkup('The ID of the Assist entity.'))
      ->setReadOnly(TRUE);

    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(new TranslatableMarkup('UUID'))
      ->setDescription(new TranslatableMarkup('The UUID of the Assist entity.'))
      ->setReadOnly(TRUE);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Administrative title'))
      ->setDescription(new TranslatableMarkup('The administrative title of the this assistant library item entity.'))
      ->setRequired(TRUE)
      ->setSettings([
        'default_value' => '',
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid']
      ->setLabel(new TranslatableMarkup('Authored by'))
      ->setDescription(new TranslatableMarkup('The username of the content author.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 97,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['status']
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => 98,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['content'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Content'))
      ->setDescription(new TranslatableMarkup('The content of the Site assistant.'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setRequired(TRUE)
      ->setSetting('target_type', 'assistant_list_entry')
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_entity_view',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'inline_entity_form_complex',
        'weight' => -1,
        'region' => 'content',
        'settings' => [
          'form_mode' => 'default',
          'revision' => TRUE,
          'override_labels' => TRUE,
          'label_singular' => 'Entry',
          'label_plural' => 'Entries',
          'collapsible' => FALSE,
          'collapsed' => FALSE,
          'allow_asymmetric_translation' => TRUE,
        ],
      ]);

    // Visibility conditions field - stores condition plugin configurations.
    $fields['visibility_plugins'] = BaseFieldDefinition::create('visibility_condition')
      ->setLabel(new TranslatableMarkup('Visibility conditions'))
      ->setDescription(new TranslatableMarkup('The visibility conditions that control when this assistant is displayed.'))
      ->setRevisionable(TRUE)
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setDisplayConfigurable('view', FALSE);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(new TranslatableMarkup('Language'))
      ->setDisplayOptions('view', [
        'region' => 'hidden',
      ])
      ->setDisplayOptions('form', [
        'type' => 'language_select',
        'weight' => 2,
      ])
      ->setDescription(new TranslatableMarkup('Language of Assistant entity.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Created'))
      ->setDescription(new TranslatableMarkup('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(new TranslatableMarkup('Changed'))
      ->setDescription(new TranslatableMarkup('The time that the entity was last edited.'));

    $fields['weight'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Weight'))
      ->setDescription(new TranslatableMarkup('The weight of this term in relation to other terms.'))
      ->setDefaultValue(0);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getWeight(): int {
    return (int) $this->get('weight')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setWeight(int $weight): static {
    $this->set('weight', $weight);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    $contexts = parent::getCacheContexts();

    foreach ($this->getVisibilityPlugins() as $plugin) {
      $contexts = Cache::mergeContexts($contexts, $plugin->getCacheContexts());
    }

    return $contexts;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    $tags = parent::getCacheTags();

    foreach ($this->getVisibilityPlugins() as $plugin) {
      $tags = Cache::mergeTags($tags, $plugin->getCacheTags());
    }

    return $tags;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getContent() {
    return $this->content;
  }

  /**
   * {@inheritdoc}
   */
  public function getVisibility(): array {
    return $this->getVisibilityConditions()->getConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function setVisibilityConfig(string $instance_id, array $configuration): static {
    $conditions = $this->getVisibilityConditions();
    if (!$conditions->has($instance_id)) {
      $configuration['id'] = $instance_id;
      $conditions->addInstanceId($instance_id, $configuration);
    }
    else {
      $conditions->setInstanceConfiguration($instance_id, $configuration);
    }
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function &getVisibilityConditions(): ConditionPluginCollection {
    if (!isset($this->visibilityCollection)) {
      $visibility_data = [];

      // Load visibility plugin configurations from the field.
      if (!$this->get('visibility_plugins')->isEmpty()) {
        foreach ($this->get('visibility_plugins') as $item) {
          $config = $item->getValue();
          if (is_array($config['value']) && isset($config['value']['id'])) {
            $visibility_data[$config['value']['id']] = $config['value'];
          }
        }
      }

      $this->visibilityCollection = new ConditionPluginCollection(
        $this->conditionPluginManager(),
        $visibility_data
      );
    }
    return $this->visibilityCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function getVisibilityCondition(string $instance_id): ConditionInterface {
    return $this->getVisibilityConditions()->get($instance_id);
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public function getVisibilityPlugins(): iterable {
    yield from $this->getVisibilityConditions()->getIterator();
  }

  /**
   * Gets the condition plugin manager.
   *
   * @return \Drupal\Core\Condition\ConditionManager
   *   The condition plugin manager.
   */
  protected function conditionPluginManager() {
    if (!isset($this->conditionPluginManager)) {
      $this->conditionPluginManager = \Drupal::service('plugin.manager.condition');
    }
    return $this->conditionPluginManager;
  }

  /**
   * {@inheritdoc}
   */
  public function getPluginCollections(): array {
    return [
      'visibility_plugins' => $this->getVisibilityConditions(),
    ];
  }

}
