<?php

namespace Drupal\site_assistant\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EditorialContentEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityViewBuilder;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\site_assistant\AssistantLibraryItemAccessControlHandler;
use Drupal\site_assistant\AssistantLibraryItemInterface;
use Drupal\site_assistant\AssistantLibraryItemListBuilder;
use Drupal\site_assistant\Form\AssistantLibraryItemForm;
use Drupal\site_assistant\Traits\EntityCreatedTrait;
use Drupal\site_assistant\Traits\SiteAssistantRevisionTrait;
use Drupal\site_assistant\Traits\SiteAssistantTranslationTrait;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the assistant library item entity.
 *
 * @ingroup site_assistant
 */
#[ContentEntityType(
  id: 'assistant_library_item',
  label : new TranslatableMarkup('Assistant library item'),
  handlers: [
    'view_builder' => EntityViewBuilder::class,
    'list_builder' => AssistantLibraryItemListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'add' => AssistantLibraryItemForm::class,
      'edit' => AssistantLibraryItemForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'access' => AssistantLibraryItemAccessControlHandler::class,
  ],
  admin_permission: 'administer site_assistant',
  base_table: 'assistant_library_item',
  data_table: 'assistant_library_item_field_data',
  revision_table: 'assistant_library_item_revision',
  revision_data_table: 'assistant_library_item_field_revision',
  show_revision_ui: TRUE,
  translatable: TRUE,
  entity_keys: [
    'id' => 'id',
    'revision' => 'vid',
    'label' => 'title',
    'langcode' => 'langcode',
    'uuid' => 'uuid',
    'status' => 'status',
    'published' => 'status',
    'uid' => 'uid',
    'owner' => 'uid',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_user',
    'revision_created' => 'revision_created',
    'revision_log_message' => 'revision_log_message',
  ],
  links: [
    'canonical' => '/site_assistant/assistant_library_item/{assistant_library_item]',
    'edit-form' => '/site_assistant/assistant_library_item/{assistant_library_item]/edit',
    'delete-form' => '/site_assistant/assistant_library_item/{assistant_library_item]/delete',
    'revision' => '/site_assistant/assistant_library_item/{assistant_library_item]/revisions/{assistant_library_item_revision]/view',
    'collection' => '/site_assistant/assistant_library_item/list',
  ],
)]
class AssistantLibraryItem extends EditorialContentEntityBase implements AssistantLibraryItemInterface {

  use EntityCreatedTrait;
  use EntityOwnerTrait;
  use SiteAssistantRevisionTrait;
  use SiteAssistantTranslationTrait;

  /**
   * {@inheritdoc}
   *
   * When a new entity instance is added, set the uid entity reference to
   * the current user as the creator of the instance.
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values): void {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);

    $this->enforceTranslationOwnerId();
    $this->setRevisionOwnerId();
  }

  /**
   * {@inheritdoc}
   *
   * Define the field properties here.
   *
   * Field name, type and size determine the table structure.
   *
   * In addition, we can define how the field and its content can be manipulated
   * in the GUI. The behavior of the widgets used can be determined here.
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    /** @var \Drupal\Core\Field\BaseFieldDefinition[] $fields */
    $fields = parent::baseFieldDefinitions($entity_type);
    $fields += static::ownerBaseFieldDefinitions($entity_type);

    // Standard field, used as unique if primary index.
    $fields['id'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('ID'))
      ->setDescription(t('The ID of the assistant library item entity.'))
      ->setReadOnly(TRUE);

    // Standard field, unique outside of the scope of the current project.
    $fields['uuid'] = BaseFieldDefinition::create('uuid')
      ->setLabel(t('UUID'))
      ->setDescription(t('The UUID of the assistant library item entity.'))
      ->setReadOnly(TRUE);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Administrative title'))
      ->setDescription(t('The administrative title of the this assistant library item entity.'))
      ->setRequired(TRUE)
      ->setSettings([
        'default_value' => '',
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -6,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid']
      ->setLabel(t('Authored by'))
      ->setDescription(t('The username of the content author.'))
      ->setRevisionable(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 97,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['status']
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => TRUE,
        ],
        'weight' => 98,
      ])
      ->setDisplayConfigurable('form', TRUE);

    $fields['content'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Content'))
      ->setDescription(t('The content of this assistant library item.'))
      ->setTranslatable(TRUE)
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'assistant_list_entry')
      ->setCardinality(FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED)
      ->setRequired(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'entity_reference_entity_view',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'inline_entity_form_complex',
        'weight' => -1,
        'region' => 'content',
        'settings' => [
          'form_mode' => 'default',
          'revision' => TRUE,
          'override_labels' => TRUE,
          'label_singular' => t('Entry'),
          'label_plural' => t('Entries'),
          'collapsible' => FALSE,
          'collapsed' => FALSE,
          'allow_asymmetric_translation' => TRUE,
        ],
      ]);

    $fields['langcode'] = BaseFieldDefinition::create('language')
      ->setLabel(t('Language'))
      ->setDisplayOptions('view', [
        'region' => 'hidden',
      ])
      ->setDisplayOptions('form', [
        'type' => 'language_select',
        'weight' => 2,
      ])
      ->setDescription(t('Language of Assistant library item.'));

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));
    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    return $fields;
  }

}
