<?php

namespace Drupal\site_assistant\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\site_assistant\Traits\SiteAssistantFormAuthorTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormGeneralTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormMetaTrait;
use Drupal\site_assistant\Traits\SiteAssistantFormOptionsTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Form controller for the site_assistant entity edit forms.
 *
 * @ingroup site_assistant
 */
class AssistantLibraryItemForm extends ContentEntityForm {

  use AutowireTrait;
  use SiteAssistantFormAuthorTrait;
  use SiteAssistantFormGeneralTrait;
  use SiteAssistantFormMetaTrait;
  use SiteAssistantFormOptionsTrait;

  /**
   * AssistantLibraryItemForm constructor.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   Entity repository.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language
   *   Language manager.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   Current user.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $dateFormatter
   *   Date formatter.
   */
  public function __construct(
    #[Autowire('entity.repository')]
    EntityRepositoryInterface $entity_repository,
    #[Autowire('entity_type.bundle.info')]
    EntityTypeBundleInfoInterface $entity_type_bundle_info,
    #[Autowire('datetime.time')]
    TimeInterface $time,
    #[Autowire('language_manager')]
    protected LanguageManagerInterface $language,
    #[Autowire('current_user')]
    protected AccountInterface $currentUser,
    #[Autowire('date.formatter')]
    protected DateFormatterInterface $dateFormatter,
  ) {
    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);
    /** @var \Drupal\site_assistant\AssistantLibraryItemInterface $entity */
    $entity = $this->entity;

    $this->addFormElementGeneral($entity, $form);
    $this->addFormElementMeta($entity, $form);
    $this->addFormElementAuthor($entity, $form);
    $this->addFormElementOptions($entity, $form);

    /** @var array{
     *   '#attached'?: array<string, array>,
     *   'advanced'?: array<string, array<string, array>>,
     *   'revision_information'?: array,
     *   'status'?: array,
     *   ...
     * } $form */
    $form['advanced']['#attributes']['class'][] = 'entity-meta';
    $form['advanced']['#type'] = 'container';

    $form['status']['#group'] = 'footer';

    $form['#attached']['library'][] = 'site_assistant/form';

    $form['revision_information']['#access'] = FALSE;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $status = parent::save($form, $form_state);

    $entity = $this->entity;
    if ($status === SAVED_UPDATED) {
      $this->messenger()->addMessage($this->t('The assistant library item %assistant_library_item has been updated.', ['%assistant_library_item' => $entity->toLink()->toString()]));
    }
    else {
      $this->messenger()->addMessage($this->t('The assistant library item %assistant_library_item has been added.', ['%assistant_library_item' => $entity->toLink()->toString()]));
    }

    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $status;
  }

}
