<?php

namespace Drupal\site_assistant\Form;

use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerTrait;

/**
 * Assistant list entry type entity form.
 *
 * @package Drupal\site_assistant\Form
 */
class AssistantListEntryTypeEntityForm extends BundleEntityFormBase {

  use MessengerTrait;

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $entity_type = $this->entity;
    $content_entity_id = $entity_type->getEntityType()->getBundleOf();

    /** @var array{
     *   label: array,
     *   id: array,
     *   ...
     * } $form */
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity_type->label(),
      '#description' => $this->t("Label for the %content_entity_id entity type (bundle).", ['%content_entity_id' => $content_entity_id]),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity_type->id(),
      '#machine_name' => [
        'exists' => '\Drupal\site_assistant\Entity\AssistantListEntryType::load',
      ],
      '#disabled' => !$entity_type->isNew(),
    ];

    return $this->protectBundleIdElement($form);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity_type = $this->entity;
    $status = $entity_type->save();
    $message_params = [
      '%label' => $entity_type->label(),
      '%content_entity_id' => $entity_type->getEntityType()->getBundleOf(),
    ];

    // Redirect to collection after saving.
    $form_state->setRedirectUrl($entity_type->toUrl('collection'));

    // Provide a message for the user.
    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label %content_entity_id entity type.', $message_params));
        return SAVED_NEW;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label %content_entity_id entity type.', $message_params));
        return SAVED_UPDATED;
    }
  }

}
