<?php

declare(strict_types=1);

namespace Drupal\site_assistant\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'visibility_condition' field type.
 */
#[FieldType(
    id: 'visibility_condition',
    label: new TranslatableMarkup('Visibility condition'),
    description: new TranslatableMarkup('Stores a single visibility condition configuration.'),
    default_formatter: 'string',
  )]
class VisibilityConditionItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties = [];
    $properties['value'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Configuration'))
      ->setRequired(TRUE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  #[\Override]
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'value' => [
          'type' => 'text',
          'size' => 'big',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  // @phpstan-ignore-next-line
  public function setValue($value, $notify = TRUE): void {
    // If the value is an array, serialize it.
    if (is_array($value) && isset($value['value']) && is_array($value['value'])) {
      $value['value'] = serialize($value['value']);
    }
    elseif (is_array($value) && !isset($value['value'])) {
      $value = ['value' => serialize($value)];
    }

    parent::setValue($value, $notify);
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(): array {
    $values = parent::getValue();

    // Unserialize the value if it's serialized.
    if (isset($values['value']) && is_string($values['value'])) {
      $unserialized = $this->unserializeValue($values['value']);
      if (!is_null($unserialized)) {
        $values['value'] = $unserialized;
      }
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $value = $this->get('value')->getValue();

    if ($value === NULL || $value === '') {
      return TRUE;
    }

    if (is_string($value)) {
      $unserialized = $this->unserializeValue($value);
      if (!is_null($unserialized)) {
        $value = $unserialized;
      }

      assert(is_array($value));
      return (count($value) === 0);
    }

    return FALSE;
  }

  /**
   * Gets the unserialized configuration array.
   *
   * @return array<string, mixed>
   *   The condition configuration.
   */
  public function getConfiguration(): array {
    $value = $this->get('value')->getValue();

    if (is_array($value)) {
      return $value;
    }

    if (!is_string($value)) {
      return [];
    }
    $unserialized = $this->unserializeValue($value);
    if (is_array($unserialized)) {
      return $unserialized;
    }

    return [];
  }

  /**
   * Unserializes the given value.
   *
   * @param string $value
   *   Serialized data.
   *
   * @return array|null
   *   Either the unserialized data or NULL on errors.
   */
  protected function unserializeValue(string $value): ?array {
    $unserialized = @unserialize($value, ['allowed_classes' => FALSE]);
    return is_array($unserialized) ? $unserialized : NULL;
  }

}
