<?php

namespace Drupal\site_assistant\Services;

use Drupal\Core\Condition\ConditionManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\Context\ContextHandlerInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Drupal\Core\Plugin\ContextAwarePluginInterface;
use Drupal\site_assistant\AssistantInterface;
use Drupal\site_assistant\SiteAssistantInterface;

/**
 * Class CustomService.
 */
class SiteAssistant implements SiteAssistantInterface {

  /**
   * SiteAssistant constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   Entity type manager.
   * @param \Drupal\Core\Plugin\Context\ContextHandlerInterface $contextHandler
   *   Context handler.
   * @param \Drupal\Core\Plugin\Context\ContextRepositoryInterface $contextRepository
   *   Context repository.
   * @param \Drupal\Core\Condition\ConditionManager $conditionManager
   *   Condition manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ContextHandlerInterface $contextHandler,
    protected ContextRepositoryInterface $contextRepository,
    protected ConditionManager $conditionManager,
  ) {
    // Nothing to do here.
  }

  /**
   * Get the active assistant.
   *
   * @return \Drupal\site_assistant\AssistantInterface|null
   *   A assist object or null if none evaluated for this request.
   */
  public function getActiveAssistant(): ?AssistantInterface {
    $active_assistants = [];
    $active_assistants_weight = [];

    foreach ($this->getAllAssistants() as $assistant) {
      if ($this->assistSectionEvaluate($assistant)) {
        $active_assistants[$assistant->id()] = $assistant;
        $active_assistants_weight[$assistant->id()] = $assistant->getWeight();
      }
    }

    // Sort active assists by weight.
    asort($active_assistants_weight);

    $active_assistant_id = array_key_first($active_assistants_weight);

    return $active_assistants[$active_assistant_id] ?? NULL;
  }

  /**
   * Get all assistants.
   *
   * @return \Drupal\site_assistant\AssistantInterface[]
   *   Array of all defined assistants.
   */
  private function getAllAssistants(): array {
    $storage = $this->entityTypeManager->getStorage('assistant');

    /** @var \Drupal\site_assistant\AssistantInterface[] $results */
    $results = $storage->loadMultiple();
    return $results;
  }

  /**
   * Evaluate all condition plugins of this assistant.
   *
   * @param \Drupal\site_assistant\AssistantInterface $assistant
   *   Assistant content.
   *
   * @return bool
   *   TRUE if all condition plugins evaluated true, FALSE if any evaluated
   *   false.
   */
  private function assistSectionEvaluate(AssistantInterface $assistant): bool {
    $visibility_plugins = $assistant->getVisibilityPlugins();

    /** @var \Drupal\Core\Condition\ConditionInterface $plugin */
    foreach ($visibility_plugins as $plugin) {
      // Inject runtime contexts if needed.
      if ($plugin instanceof ContextAwarePluginInterface) {
        /** @var string[] $context_mapping */
        $context_mapping = $plugin->getContextMapping();
        $contexts = $this->contextRepository->getRuntimeContexts($context_mapping);
        $this->contextHandler->applyContextMapping($plugin, $contexts);
      }

      $result = $this->conditionManager->execute($plugin);
      if ($result === FALSE) {
        // No need to process any other plugin.
        return FALSE;
      }
    }

    return TRUE;
  }

}
