<?php

namespace Drupal\site_assistant\Traits;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\RevisionLogInterface;

/**
 * Trait for revisionable classes.
 */
trait SiteAssistantRevisionTrait {

  /**
   * Acts on a revision before it gets saved.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage object.
   * @param \stdClass $record
   *   The revision object.
   */
  public function preSaveRevision(EntityStorageInterface $storage, \stdClass $record): void {
    parent::preSaveRevision($storage, $record);

    if (!($record instanceof RevisionLogInterface)) {
      return;
    }
    $log_message = $record->getRevisionLogMessage();
    if (!$this->isNewRevision() && isset($this->original) && ($this->original instanceof RevisionLogInterface) && ($log_message === '')) {
      // If we are updating an existing assist list entry without adding a new
      // revision, we need to make sure $entity->revision_log
      // is reset whenever it is empty.
      // Therefore, this code allows us to avoid clobbering an existing log
      // entry with an empty one.
      // @phpstan-ignore-next-line
      $record->setRevisionLogMessage($this->original->getRevisionLogMessage());
    }
  }

  /**
   * Sets the revision owner ID based on the entity owner.
   */
  protected function setRevisionOwnerId(): void {
    // If no revision author has been set explicitly, make the entity owner the
    // revision author.
    $owner_id = $this->getOwnerId();
    if (is_null($this->getRevisionUser()) && !is_null($owner_id)) {
      $this->setRevisionUserId($owner_id);
    }
  }

}
