<?php

namespace Drupal\Tests\site_assistant\Traits;

use Drupal\user\UserInterface;

/**
 * A login helper trait for a Drupal site.
 *
 * @group site_assistant
 */
trait LoginTrait {

  /**
   * Login as user with specified permissions.
   *
   * @param string|string[] $permissions
   *   List of permissions or a single permission.
   *
   * @return \Drupal\user\UserInterface
   *   A fully loaded user object with the specified permissions.
   */
  protected function loginWithPermissions($permissions = []): UserInterface {
    $permissions = (array) $permissions;
    $account = $this->createUser(
      permissions: $permissions,
    );
    $this::assertInstanceOf(UserInterface::class, $account);
    $this->drupalLogin($account);

    return $account;
  }

  /**
   * Login a user by a given role.
   *
   * Login a user during a test by passing a single role or an array of roles.
   * A temporary user will be created with the matching role(s).
   *
   * @param string[]|string $role
   *   The target role(s) for the user.
   *
   * @return \Drupal\user\UserInterface
   *   A fully loaded user object for the logged in user.
   */
  protected function loginByRole($role): UserInterface {
    $role = (array) $role;
    $account = $this->createUser([], NULL, FALSE, [
      'roles' => $role,
    ]);
    $this::assertInstanceOf(UserInterface::class, $account);
    $this->drupalLogin($account);

    return $account;
  }

  /**
   * Login a user with the 'administrator' role.
   *
   * @return \Drupal\user\UserInterface
   *   A fully loaded user object for the logged in user.
   */
  protected function loginAsAdministrator(): UserInterface {
    return $this->loginByRole([
      'administrator',
    ]);
  }

}
