<?php

namespace Drupal\site_audit_checklist\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Url;
use Drupal\Core\Routing\RedirectDestinationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to update a single checklist task.
 */
class TaskForm extends FormBase {

  use RedirectDestinationTrait;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new TaskForm.
   *
   * @param \Drupal\Core\Database\Connection $database
   * The database connection.
   */
  public function __construct(Connection $database) {
    $this->database = $database;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'site_audit_checklist_task_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $task_id = NULL) {
    // Get the current task's status from the database.
    $record = $this->database->select('site_audit_checklist_status', 's')
      ->fields('s')
      ->condition('task_id', $task_id)
      ->execute()
      ->fetchAssoc();
    
    $form['title'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Title'),
    ];
    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
    ];

    $form['task_id'] = [
      '#type' => 'hidden',
      '#value' => $task_id,
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Mark as completed'),
      '#default_value' => isset($record['status']) ? $record['status'] : 0,
    ];
 

    $form['notes'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Notes'),
      '#default_value' => isset($record['notes']) ? $record['notes'] : '',
    ];

    $form['actions']['#type'] = 'actions';
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $task_id = $form_state->getValue('task_id');
    $status = $form_state->getValue('status');
    $notes = $form_state->getValue('notes');
    $title = $form_state->getValue('title');
    $description = $form_state->getValue('description');

    $current_user = \Drupal::currentUser();

    // Check if a record for this task already exists.
    $existing = $this->database->select('site_audit_checklist_status', 's')
      ->fields('s')
      ->condition('task_id', $task_id)
      ->execute()
      ->fetchAssoc();

    $fields = [
      'task_id' => $task_id,
      'status' => (int) $status,
      'notes' => $notes,
      'title' => $title,
      'description' => $description
    ];

    if ($status) {
      $fields['completed_by'] = $current_user->id();
      $fields['completed_timestamp'] = \Drupal::time()->getRequestTime();
    }

    if ($existing) {
      $this->database->update('site_audit_checklist_status')
        ->fields($fields)
        ->condition('task_id', $task_id)
        ->execute();
    }
    else {
      $this->database->insert('site_audit_checklist_status')
        ->fields($fields)
        ->execute();
    }

    $this->messenger()->addStatus($this->t('Checklist item updated successfully.'));
    $form_state->setRedirectUrl(Url::fromRoute('site_audit_checklist.dashboard'));
  }
}