<?php

namespace Drupal\site_health\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\site_health\Service\QueryAnalyzerService;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for miniOrange Query Monitor.
 */
class QueryMonitorController extends ControllerBase
{

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The query analyzer service.
   *
   * @var \Drupal\site_health\Service\QueryAnalyzerService
   */
  protected $queryAnalyzer;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The pager manager service.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected $pagerManager;

  /**
   * Constructs a QueryMonitorController object.
   */
  public function __construct(
    Connection $database,
    ConfigFactoryInterface $config_factory,
    FormBuilderInterface $form_builder,
    QueryAnalyzerService $query_analyzer,
    DateFormatterInterface $date_formatter,
    PagerManagerInterface $pager_manager
  ) {
    $this->database = $database;
    $this->configFactory = $config_factory;
    $this->formBuilder = $form_builder;
    $this->queryAnalyzer = $query_analyzer;
    $this->dateFormatter = $date_formatter;
    $this->pagerManager = $pager_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('database'),
      $container->get('config.factory'),
      $container->get('form_builder'),
      $container->get('site_health.query_analyzer'),
      $container->get('date.formatter'),
      $container->get('pager.manager')
    );
  }

  /**
   * Admin configuration page.
   */
  public function adminConfig()
  {
    return $this->formBuilder->getForm('Drupal\site_health\Form\QueryMonitorConfigForm');
  }

  /**
   * Main query report page.
   */
  public function queryReport()
  {
    $build = [];

    $build['overview'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-monitor-overview']],
    ];

    $build['overview']['stats'] = $this->getOverviewStats();

    // Add delete all queries button
    $build['overview']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-monitor-actions']],
      '#weight' => 10,
    ];

    $build['overview']['actions']['export_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Export All Data'),
      '#url' => \Drupal\Core\Url::fromRoute('site_health.export_all_queries'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'style' => 'margin-top: 20px;',
        'target' => '_blank',
      ],
      '#access' => $this->currentUser()->hasPermission('export query monitor data'),
    ];

    $build['overview']['actions']['delete_all'] = [
      '#type' => 'link',
      '#title' => $this->t('Delete All Queries'),
      '#url' => \Drupal\Core\Url::fromRoute('site_health.delete_all_queries'),
      '#attributes' => [
        'class' => ['button', 'button--danger'],
        'style' => 'margin-top: 20px;margin-right: 10px;',
      ],
      '#access' => $this->currentUser()->hasPermission('delete query monitor logs'),
    ];

    // Attach the CSS library for styling.
    $build['#attached']['library'][] = 'site_health/site_health.admin';

    return $build;
  }

  /**
   * Slow queries report.
   */
  public function slowQueries()
  {
    $config = $this->configFactory->get('site_health.settings');
    $threshold = $config->get('slow_query_threshold') ?: 1000;

    // Set up pagination.
    $limit = 25; // Items per page
    $page = $this->pagerManager->createPager(
      $this->queryAnalyzer->getSlowQueriesCount($threshold),
      $limit
    )->getCurrentPage();
    $offset = $page * $limit;

    $queries = $this->queryAnalyzer->getSlowQueries($threshold, $limit, $offset);

    $build['export_actions']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-monitor-category-actions']],
      '#weight' => 10,
    ];

    $build['export_actions']['actions']['export_csv'] = [
      '#type' => 'link',
      '#title' => $this->t('Export as CSV'),
      '#url' => \Drupal\Core\Url::fromRoute('site_health.export_slow_queries'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
      '#access' => $this->currentUser()->hasPermission('export query monitor data'),
    ];

    $build['query_data'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Query'),
        $this->t('Avg Execution Time (ms)'),
        $this->t('Max Execution Time (ms)'),
        $this->t('Count'),
        $this->t('Last Executed'),
      ],
      '#rows' => $this->formatQueryData($queries),
      '#empty' => $this->t('No slow queries found.'),
      '#attached' => [
        'library' => ['site_health/site_health.admin'],
      ],
    ];

    // Add pagination.
    $build['pager'] = [
      '#type' => 'pager',
      '#weight' => 20,
    ];

    return $build;
  }

  /**
   * Heavy queries report.
   */
  public function heavyQueries()
  {
    $config = $this->configFactory->get('site_health.settings');
    $threshold = $config->get('heavy_query_threshold') ?: 100;

    // Set up pagination.
    $limit = 25; // Items per page
    $page = $this->pagerManager->createPager(
      $this->queryAnalyzer->getHeavyQueriesCount($threshold),
      $limit
    )->getCurrentPage();
    $offset = $page * $limit;

    $queries = $this->queryAnalyzer->getHeavyQueries($threshold, $limit, $offset);

    $build['export_actions']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-monitor-category-actions']],
      '#weight' => 10,
    ];

    $build['export_actions']['actions']['export_csv'] = [
      '#type' => 'link',
      '#title' => $this->t('Export as CSV'),
      '#url' => \Drupal\Core\Url::fromRoute('site_health.export_heavy_queries'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
      '#access' => $this->currentUser()->hasPermission('export query monitor data'),
    ];

    $build['query_data'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Query'),
        $this->t('Avg Memory Usage (KB)'),
        $this->t('Max Memory Usage (KB)'),
        $this->t('Count'),
        $this->t('Last Executed'),
      ],
      '#rows' => $this->formatQueryData($queries, 'memory'),
      '#empty' => $this->t('No heavy queries found.'),
      '#attached' => [
        'library' => ['site_health/site_health.admin'],
      ],
    ];

    // Add pagination.
    $build['pager'] = [
      '#type' => 'pager',
      '#weight' => 20,
    ];

    return $build;
  }

  /**
   * Frequent queries report.
   */
  public function frequentQueries()
  {
    // Set up pagination.
    $limit = 25; // Items per page
    $page = $this->pagerManager->createPager(
      $this->queryAnalyzer->getFrequentQueriesCount(),
      $limit
    )->getCurrentPage();
    $offset = $page * $limit;

    $queries = $this->queryAnalyzer->getFrequentQueries($limit, $offset);

    $build['export_actions']['actions'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-monitor-category-actions']],
      '#weight' => 10,
    ];
    $build['export_actions']['actions']['export_csv'] = [
      '#type' => 'link',
      '#title' => $this->t('Export as CSV'),
      '#url' => \Drupal\Core\Url::fromRoute('site_health.export_frequent_queries'),
      '#attributes' => [
        'class' => ['button', 'button--primary'],
        'target' => '_blank',
      ],
      '#access' => $this->currentUser()->hasPermission('export query monitor data'),
    ];
    $build['query_data'] = [
      '#theme' => 'table',
      '#header' => [
        $this->t('Query'),
        $this->t('Count'),
        $this->t('Avg Execution Time (ms)'),
        $this->t('Total Time (ms)'),
        $this->t('Last Executed'),
      ],
      '#rows' => $this->formatQueryData($queries, 'frequency'),
      '#empty' => $this->t('No frequent queries found.'),
      '#attached' => [
        'library' => ['site_health/site_health.admin'],
      ],
    ];

    // Add pagination.
    $build['pager'] = [
      '#type' => 'pager',
      '#weight' => 20,
    ];

    return $build;
  }

  /**
   * Get overview statistics as render array.
   */
  private function getOverviewStats()
  {
    $stats = $this->queryAnalyzer->getOverviewStats();

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['query-stats']],
      'total_queries' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['stat-item']],
        'value' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#value' => number_format($stats['total_queries']),
        ],
        'label' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('Total Queries'),
        ],
      ],
      'slow_queries' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['stat-item']],
        'value' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#value' => number_format($stats['slow_queries']),
        ],
        'label' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('Slow Queries'),
        ],
      ],
      'avg_execution_time' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['stat-item']],
        'value' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#value' => number_format($stats['avg_execution_time'], 2) . 'ms',
        ],
        'label' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('Avg Execution Time'),
        ],
      ],
      'total_memory' => [
        '#type' => 'container',
        '#attributes' => ['class' => ['stat-item']],
        'value' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#value' => number_format($stats['total_memory'] / (1024 * 1024), 2) . 'MB',
        ],
        'label' => [
          '#type' => 'html_tag',
          '#tag' => 'p',
          '#value' => $this->t('Total Memory Used'),
        ],
      ],
    ];
  }

  /**
   * Format query data for table display.
   */
  private function formatQueryData($queries, $type = 'time')
  {
    $rows = [];

    foreach ($queries as $query) {
      $truncated_query = strlen($query->query) > 100 ? substr($query->query, 0, 100) . '...' : $query->query;

      // Sanitize the query for safe display
      $safe_query = [
        '#type' => 'html_tag',
        '#tag' => 'code',
        '#value' => $truncated_query,
        '#attributes' => [
          'class' => ['query-display'],
          'title' => $this->t('Click to view full query'),
        ],
      ];

      switch ($type) {
        case 'memory':
          $rows[] = [
            ['data' => $safe_query],
            number_format($query->avg_memory / 1024, 2),
            number_format($query->max_memory / 1024, 2),
            number_format($query->count),
            $this->dateFormatter->format($query->last_executed, 'short'),
          ];
          break;

        case 'frequency':
          $rows[] = [
            ['data' => $safe_query],
            number_format($query->count),
            number_format($query->avg_execution_time, 2),
            number_format($query->total_time, 2),
            $this->dateFormatter->format($query->last_executed, 'short'),
          ];
          break;

        default: // time
          $rows[] = [
            ['data' => $safe_query],
            number_format($query->avg_execution_time, 2),
            number_format($query->max_execution_time, 2),
            number_format($query->count),
            $this->dateFormatter->format($query->last_executed, 'short'),
          ];
          break;
      }
    }

    return $rows;
  }

  /**
   * Export slow queries as CSV.
   */
  public function exportSlowQueries()
  {
    $config = $this->configFactory->get('site_health.settings');
    $threshold = $config->get('slow_query_threshold') ?: 1000;

    $queries = $this->queryAnalyzer->getSlowQueries($threshold, 1000, 0); // Export up to 1000 records from the beginning

    $headers = [
      'Query Hash',
      'Query',
      'Average Execution Time (ms)',
      'Maximum Execution Time (ms)',
      'Execution Count',
      'Last Executed',
    ];

    return $this->generateCsvResponse($queries, $headers, 'slow-queries', 'time');
  }

  /**
   * Export heavy queries as CSV.
   */
  public function exportHeavyQueries()
  {
    $config = $this->configFactory->get('site_health.settings');
    $threshold = $config->get('heavy_query_threshold') ?: 100;

    $queries = $this->queryAnalyzer->getHeavyQueries($threshold, 1000, 0);

    $headers = [
      'Query Hash',
      'Query',
      'Average Memory Usage (KB)',
      'Maximum Memory Usage (KB)',
      'Execution Count',
      'Last Executed',
    ];

    return $this->generateCsvResponse($queries, $headers, 'heavy-queries', 'memory');
  }

  /**
   * Export frequent queries as CSV.
   */
  public function exportFrequentQueries()
  {
    $queries = $this->queryAnalyzer->getFrequentQueries(1000, 0);

    $headers = [
      'Query Hash',
      'Query',
      'Execution Count',
      'Average Execution Time (ms)',
      'Total Time (ms)',
      'Last Executed',
    ];

    return $this->generateCsvResponse($queries, $headers, 'frequent-queries', 'frequency');
  }

  /**
   * Export all queries as CSV.
   */
  public function exportAllQueries()
  {
    // Get all queries from the database with a reasonable limit
    $query = $this->database->select('site_health_query_log', 'q')
      ->fields('q')
      ->orderBy('timestamp', 'DESC')
      ->range(0, 5000); // Limit to 5000 most recent queries

    $queries = $query->execute()->fetchAll();

    $headers = [
      'ID',
      'Query Hash',
      'Query',
      'Execution Time (ms)',
      'Memory Usage (bytes)',
      'Timestamp',
      'URI',
      'Caller',
    ];

    return $this->generateCsvResponseRaw($queries, $headers, 'all-queries');
  }

  /**
   * Generate CSV response for formatted query data.
   */
  private function generateCsvResponse($queries, $headers, $filename_prefix, $type = 'time')
  {
    $csv_data = [];
    $csv_data[] = $headers;

    foreach ($queries as $query) {
      switch ($type) {
        case 'memory':
          $csv_data[] = [
            $query->query_hash,
            $this->sanitizeForCsv($query->query),
            number_format($query->avg_memory / 1024, 2),
            number_format($query->max_memory / 1024, 2),
            $query->count,
            $this->dateFormatter->format($query->last_executed, 'custom', 'Y-m-d H:i:s'),
          ];
          break;

        case 'frequency':
          $csv_data[] = [
            $query->query_hash,
            $this->sanitizeForCsv($query->query),
            $query->count,
            number_format($query->avg_execution_time, 2),
            number_format($query->total_time, 2),
            $this->dateFormatter->format($query->last_executed, 'custom', 'Y-m-d H:i:s'),
          ];
          break;

        default: // time
          $csv_data[] = [
            $query->query_hash,
            $this->sanitizeForCsv($query->query),
            number_format($query->avg_execution_time, 2),
            number_format($query->max_execution_time, 2),
            $query->count,
            $this->dateFormatter->format($query->last_executed, 'custom', 'Y-m-d H:i:s'),
          ];
          break;
      }
    }

    return $this->createCsvDownload($csv_data, $filename_prefix);
  }

  /**
   * Generate CSV response for raw query data.
   */
  private function generateCsvResponseRaw($queries, $headers, $filename_prefix)
  {
    $csv_data = [];
    $csv_data[] = $headers;

    foreach ($queries as $query) {
      $csv_data[] = [
        $query->id,
        $query->query_hash,
        $this->sanitizeForCsv($query->query),
        $query->execution_time,
        $query->memory_usage,
        $this->dateFormatter->format($query->timestamp, 'custom', 'Y-m-d H:i:s'),
        $this->sanitizeForCsv($query->uri),
        $this->sanitizeForCsv($query->caller),
      ];
    }

    return $this->createCsvDownload($csv_data, $filename_prefix);
  }

  /**
   * Create CSV download response.
   */
  private function createCsvDownload($csv_data, $filename_prefix)
  {
    $csv_content = '';
    foreach ($csv_data as $row) {
      $csv_content .= '"' . implode('","', $row) . '"' . "\n";
    }

    $filename = $filename_prefix . '-' . date('Y-m-d-H-i-s') . '.csv';

    $response = new Response($csv_content);
    $response->headers->set('Content-Type', 'text/csv');
    $response->headers->set('Content-Disposition', 'attachment; filename="' . $filename . '"');
    $response->headers->set('Cache-Control', 'private, max-age=0, must-revalidate');
    $response->headers->set('Pragma', 'public');

    return $response;
  }

  /**
   * Sanitize data for CSV export.
   */
  private function sanitizeForCsv($value)
  {
    // Remove newlines and carriage returns that could break CSV
    $value = str_replace(["\r", "\n", "\t"], [' ', ' ', ' '], $value);
    // Escape double quotes by doubling them
    $value = str_replace('"', '""', $value);
    return $value;
  }
}
