<?php

namespace Drupal\site_health\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\site_health\Service\QueryAnalyzerService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Provides a confirmation form for deleting all query logs.
 */
class DeleteAllQueriesForm extends ConfirmFormBase
{

  /**
   * The query analyzer service.
   *
   * @var \Drupal\site_health\Service\QueryAnalyzerService
   */
  protected $queryAnalyzer;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a DeleteAllQueriesForm object.
   */
  public function __construct(QueryAnalyzerService $query_analyzer, LoggerInterface $logger)
  {
    $this->queryAnalyzer = $query_analyzer;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('site_health.query_analyzer'),
      $container->get('logger.factory')->get('site_health')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'site_health_delete_all_queries_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion()
  {
    return $this->t('Are you sure you want to delete all query logs?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription()
  {
    return $this->t('This action cannot be undone. All query monitoring data will be permanently removed. Only users with "Delete Query Monitor Logs" permission can perform this action.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl()
  {
    return new Url('site_health.query_report');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText()
  {
    return $this->t('Delete All Queries');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    // Double-check permissions before deletion
    if (!\Drupal::currentUser()->hasPermission('delete query monitor logs')) {
      $this->messenger()->addError($this->t('Access denied. You do not have permission to delete query logs.'));
      $form_state->setRedirectUrl($this->getCancelUrl());
      return;
    }

    try {
      $deleted_count = $this->queryAnalyzer->deleteAllQueries();

      if ($deleted_count > 0) {
        $this->messenger()->addMessage(
          $this->t('Successfully deleted @count query logs.', ['@count' => number_format($deleted_count)])
        );

        // Log the deletion for audit purposes
        $this->logger->notice(
          'User @user deleted @count query logs.',
          [
            '@user' => \Drupal::currentUser()->getAccountName(),
            '@count' => $deleted_count,
          ]
        );
      } else {
        $this->messenger()->addMessage($this->t('No query logs were found to delete.'));
      }
    } catch (\Exception $e) {
      $this->messenger()->addError(
        $this->t('An error occurred while deleting query logs: @error', ['@error' => $e->getMessage()])
      );

      $this->logger->error(
        'Error deleting query logs: @error',
        ['@error' => $e->getMessage()]
      );
    }

    $form_state->setRedirectUrl($this->getCancelUrl());
  }
}
