<?php

namespace Drupal\site_health\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for miniOrange Query Monitor.
 */
class QueryMonitorConfigForm extends ConfigFormBase
{

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['site_health.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'site_health_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('site_health.settings');

    // Remove vertical tabs, show sections one after another.

    // Monitoring Settings.
    $form['enable_monitoring'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Query Monitoring'),
      '#description' => $this->t('Enable database query monitoring and logging. <strong>Warning:</strong> This may impact performance on high-traffic sites.'),
      '#default_value' => $config->get('enable_monitoring'),
    ];

    $form['monitoring'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Monitoring Settings'),
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['monitoring']['sampling_rate'] = [
      '#type' => 'number',
      '#title' => $this->t('Sampling Rate (%)'),
      '#description' => $this->t('Percentage of requests to monitor (1-100). Lower values reduce performance impact.'),
      '#default_value' => $config->get('sampling_rate') ?: 10,
      '#min' => 1,
      '#max' => 100,
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['monitoring']['rate_limit_per_minute'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate Limit (requests per minute)'),
      '#description' => $this->t('Maximum number of requests to log per minute to prevent excessive logging.'),
      '#default_value' => $config->get('rate_limit_per_minute') ?: 60,
      '#min' => 1,
      '#max' => 1000,
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Security Settings.
    $form['security'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Security Settings'),
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['security']['sanitize_queries'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Sanitize Sensitive Data'),
      '#description' => $this->t('Remove passwords, tokens, and other sensitive data from logged queries.'),
      '#default_value' => $config->get('sanitize_queries') !== FALSE,
    ];

    $form['security']['log_caller_info'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log Caller Information'),
      '#description' => $this->t('Include information about which code called the query. May expose application structure.'),
      '#default_value' => $config->get('log_caller_info') ?: FALSE,
    ];

    $form['security']['max_query_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Query Length'),
      '#description' => $this->t('Maximum length of queries to store (characters). Longer queries will be truncated.'),
      '#default_value' => $config->get('max_query_length') ?: 2000,
      '#min' => 10,
      '#max' => 10000,
    ];

    // Query Thresholds.
    $form['thresholds'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Query Thresholds'),
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['thresholds']['slow_query_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Slow Query Threshold (ms)'),
      '#description' => $this->t('Queries taking longer than this will be marked as slow.'),
      '#default_value' => $config->get('slow_query_threshold') ?: 1000,
      '#min' => 1,
    ];

    $form['thresholds']['heavy_query_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Heavy Query Memory Threshold (KB)'),
      '#description' => $this->t('Queries using more memory than this will be marked as heavy.'),
      '#default_value' => $config->get('heavy_query_threshold') ?: 100,
      '#min' => 1,
    ];

    // Exclusion Settings.
    $form['exclusions'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Exclusion Settings'),
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $default_excluded_tables = [
      'cache_bootstrap',
      'cache_config',
      'cache_container',
      'cache_data',
      'cache_default',
      'cache_discovery',
      'cache_dynamic_page_cache',
      'cache_entity',
      'cache_menu',
      'cache_page',
      'cache_render',
      'cache_toolbar',
      'cache_rest',
      'cache_form',
      'cache_field',
      'site_health_query_log',
      'sessions',
      'watchdog',
    ];

    $form['exclusions']['excluded_tables'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Excluded Tables'),
      '#description' => $this->t('Enter table names separated by semicolons (;) to exclude from query logging. Cache tables are excluded by default for performance reasons. The query monitor table itself is always excluded to prevent infinite loops.'),
      '#default_value' => $config->get('excluded_tables') ?: implode(';', $default_excluded_tables),
      '#rows' => 6,
    ];

    $form['exclusions']['exclude_admin_paths'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Exclude Admin Paths'),
      '#description' => $this->t('Exclude queries from admin pages (/admin/*) to reduce logging volume.'),
      '#default_value' => $config->get('exclude_admin_paths') ?: FALSE,
    ];

    // Data Storage.
    $form['storage'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Data Storage'),
      '#states' => [
        'visible' => [
          ':input[name="enable_monitoring"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['storage']['log_retention_days'] = [
      '#type' => 'number',
      '#title' => $this->t('Log Retention (days)'),
      '#description' => $this->t('How many days to keep query logs. Shorter retention improves performance.'),
      '#default_value' => $config->get('log_retention_days') ?: 7,
      '#min' => 1,
      '#max' => 365,
    ];

    $form['storage']['max_log_entries'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Log Entries'),
      '#description' => $this->t('Maximum number of log entries to keep. Lower values improve performance.'),
      '#default_value' => $config->get('max_log_entries') ?: 10000,
      '#min' => 100,
      '#max' => 1000000,
    ];

    $form['storage']['use_async_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use Asynchronous Logging'),
      '#description' => $this->t('Process query logs in background using Drupal queues. Recommended for performance.'),
      '#default_value' => $config->get('use_async_logging') !== FALSE,
    ];

    // Performance warning (always visible).
    $form['performance_warning'] = [
      '#type' => 'details',
      '#title' => $this->t('⚠️ Performance Notice'),
      '#open' => FALSE,
      '#weight' => -10,
    ];

    $form['performance_warning']['warning'] = [
      '#theme' => 'status_messages',
      '#message_list' => [
        'warning' => [
          $this->t('Query monitoring can significantly impact site performance. It is recommended to:'),
          $this->t('1. Use only in development/staging environments'),
          $this->t('2. Keep sampling rate low (10% or less) for production'),
          $this->t('3. Enable asynchronous logging'),
          $this->t('4. Set appropriate exclusions for high-volume tables'),
          $this->t('5. Monitor disk usage as logs can grow quickly'),
        ]
      ],
      '#status_headings' => [
        'warning' => $this->t('Warning message'),
      ],

    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('site_health.settings');
    $clean_values = $form_state->cleanValues()->getValues();
    $config->setData($clean_values)->save();

    parent::submitForm($form, $form_state);
  }
}
