<?php

namespace Drupal\site_health\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Database;
use Drupal\site_health\Service\QueryLoggerService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for query monitoring.
 */
class QueryMonitorSubscriber implements EventSubscriberInterface
{

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The query logger service.
   *
   * @var \Drupal\site_health\Service\QueryLoggerService
   */
  protected $queryLogger;

  /**
   * Constructs a QueryMonitorSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\site_health\Service\QueryLoggerService $query_logger
   *   The query logger service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, QueryLoggerService $query_logger)
  {
    $this->configFactory = $config_factory;
    $this->queryLogger = $query_logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents()
  {
    // Listen to kernel request and response events.
    return [
      KernelEvents::REQUEST => ['onRequest', 100],
      KernelEvents::RESPONSE => ['onResponse', -100],
    ];
  }

  /**
   * Handles the kernel request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event)
  {
    // Only process the main request, not sub-requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $config = $this->configFactory->get('site_health.settings');

    // Check if monitoring is enabled
    if (!$config->get('enable_monitoring')) {
      return;
    }

    // Check if we should exclude admin paths
    if ($config->get('exclude_admin_paths')) {
      $request = $event->getRequest();
      $path = $request->getPathInfo();

      // Exclude admin paths, AJAX calls, and API endpoints for performance
      if (
        strpos($path, '/admin') === 0 ||
        strpos($path, '/batch') === 0 ||
        strpos($path, '/cron') === 0 ||
        $request->isXmlHttpRequest() ||
        strpos($path, '/system/ajax') !== FALSE ||
        strpos($path, '/quickedit/') !== FALSE
      ) {
        return;
      }
    }

    // Additional security: Don't log queries for sensitive paths
    $request = $event->getRequest();
    $path = $request->getPathInfo();
    $sensitive_paths = [
      '/user/login',
      '/user/password',
      '/user/register',
      '/admin/config/development/site-health', // Don't log our own config
    ];

    foreach ($sensitive_paths as $sensitive_path) {
      if (strpos($path, $sensitive_path) === 0) {
        return;
      }
    }

    // Start query logging for this request
    try {
      Database::startLog('site_health');
    } catch (\Exception $e) {
      // Fail silently to avoid breaking the site
      \Drupal::logger('site_health')
        ->error('Failed to start query logging: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Handles the kernel response event.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event)
  {
    // Only process the main request, not sub-requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $config = $this->configFactory->get('site_health.settings');

    if (!$config->get('enable_monitoring')) {
      return;
    }

    // Only log queries if we actually started logging
    //if (Database::getLog('miniorange_query_monitor') !== null) {
    try {
      // Log queries for this page request.
      $this->queryLogger->logPageQueries();
    } catch (\Exception $e) {
      // Fail silently to avoid breaking the site
      \Drupal::logger('site_health')
        ->error('Failed to log page queries: @message', ['@message' => $e->getMessage()]);
    }
    //}
  }
}
