<?php

namespace Drupal\site_health\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for analyzing database queries.
 */
class QueryAnalyzerService {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a QueryAnalyzerService object.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Get slow queries with pagination support.
   */
  public function getSlowQueries($threshold = 1000, $limit = 50, $offset = 0) {
    $query = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash', 'query'])
      ->condition('execution_time', $threshold, '>=')
      ->groupBy('query_hash')
      ->groupBy('query')
      ->range($offset, $limit);
    
    $query->addExpression('AVG(execution_time)', 'avg_execution_time');
    $query->addExpression('MAX(execution_time)', 'max_execution_time');
    $query->addExpression('COUNT(*)', 'count');
    $query->addExpression('MAX(timestamp)', 'last_executed');
    $query->orderBy('avg_execution_time', 'DESC');
    
    return $query->execute()->fetchAll();
  }

  /**
   * Get count of slow queries for pagination.
   */
  public function getSlowQueriesCount($threshold = 1000) {
    $subquery = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash'])
      ->condition('execution_time', $threshold, '>=')
      ->groupBy('query_hash');
    
    $count_query = $this->database->select($subquery, 'sq');
    $count_query->addExpression('COUNT(*)');
    
    return $count_query->execute()->fetchField();
  }

  /**
   * Get heavy queries (high memory usage) with pagination support.
   */
  public function getHeavyQueries($threshold = 100, $limit = 50, $offset = 0) {
    $threshold_bytes = $threshold * 1024; // Convert KB to bytes
    
    $query = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash', 'query'])
      ->condition('memory_usage', $threshold_bytes, '>=')
      ->groupBy('query_hash')
      ->groupBy('query')
      ->range($offset, $limit);
    
    $query->addExpression('AVG(memory_usage)', 'avg_memory');
    $query->addExpression('MAX(memory_usage)', 'max_memory');
    $query->addExpression('COUNT(*)', 'count');
    $query->addExpression('MAX(timestamp)', 'last_executed');
    $query->orderBy('avg_memory', 'DESC');
    
    return $query->execute()->fetchAll();
  }

  /**
   * Get count of heavy queries for pagination.
   */
  public function getHeavyQueriesCount($threshold = 100) {
    $threshold_bytes = $threshold * 1024; // Convert KB to bytes
    
    $subquery = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash'])
      ->condition('memory_usage', $threshold_bytes, '>=')
      ->groupBy('query_hash');
    
    $count_query = $this->database->select($subquery, 'sq');
    $count_query->addExpression('COUNT(*)');
    
    return $count_query->execute()->fetchField();
  }

  /**
   * Get frequently executed queries with pagination support.
   */
  public function getFrequentQueries($limit = 50, $offset = 0) {
    $query = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash', 'query'])
      ->groupBy('query_hash')
      ->groupBy('query')
      ->range($offset, $limit);
    
    $query->addExpression('COUNT(*)', 'count');
    $query->addExpression('AVG(execution_time)', 'avg_execution_time');
    $query->addExpression('SUM(execution_time)', 'total_time');
    $query->addExpression('MAX(timestamp)', 'last_executed');
    $query->orderBy('count', 'DESC');
    
    return $query->execute()->fetchAll();
  }

  /**
   * Get count of frequent queries for pagination.
   */
  public function getFrequentQueriesCount() {
    $subquery = $this->database->select('site_health_query_log', 'q')
      ->fields('q', ['query_hash'])
      ->groupBy('query_hash');
    
    $count_query = $this->database->select($subquery, 'sq');
    $count_query->addExpression('COUNT(*)');
    
    return $count_query->execute()->fetchField();
  }

  /**
   * Get overview statistics.
   */
  public function getOverviewStats() {
    $config = $this->configFactory->get('site_health.settings');
    $slow_threshold = $config->get('slow_query_threshold') ?: 1000;
    
    // Total queries
    $total_queries = $this->database->select('site_health_query_log', 'q')
      ->countQuery()
      ->execute()
      ->fetchField();
    
    // Slow queries count
    $slow_queries = $this->database->select('site_health_query_log', 'q')
      ->condition('execution_time', $slow_threshold, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();
    
    // Average execution time
    $avg_query = $this->database->select('site_health_query_log', 'q');
    $avg_query->addExpression('AVG(execution_time)', 'avg_time');
    $avg_execution_time = $avg_query->execute()->fetchField();
    
    // Total memory usage
    $memory_query = $this->database->select('site_health_query_log', 'q');
    $memory_query->addExpression('SUM(memory_usage)', 'total_memory');
    $total_memory = $memory_query->execute()->fetchField();
    
    return [
      'total_queries' => $total_queries ?: 0,
      'slow_queries' => $slow_queries ?: 0,
      'avg_execution_time' => $avg_execution_time ?: 0,
      'total_memory' => $total_memory ?: 0,
    ];
  }

  /**
   * Delete all query logs.
   *
   * @return int
   *   The number of deleted records.
   */
  public function deleteAllQueries() {
    return $this->database->delete('site_health_query_log')
      ->execute();
  }

} 