<?php

namespace Drupal\site_health\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Database\Database;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service for logging database queries with security and performance improvements.
 */
class QueryLoggerService
{

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Sensitive data patterns to remove from queries.
   *
   * @var array
   */
  protected $sensitivePatterns = [
    '/password\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/pass\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/token\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/api_key\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/secret\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/hash\s*=\s*[\'"][^\'"]*[\'"]/i',
    '/salt\s*=\s*[\'"][^\'"]*[\'"]/i',
  ];

  /**
   * Constructs a QueryLoggerService object.
   */
  public function __construct(
    Connection $database,
    ConfigFactoryInterface $config_factory,
    TimeInterface $time,
    QueueFactory $queue_factory,
    LoggerChannelFactoryInterface $logger_factory,
    StateInterface $state
  ) {
    $this->database = $database;
    $this->configFactory = $config_factory;
    $this->time = $time;
    $this->queueFactory = $queue_factory;
    $this->loggerFactory = $logger_factory;
    $this->state = $state;
  }

  /**
   * Log queries for the current page request with rate limiting and sampling.
   */
  public function logPageQueries()
  {
    try {
      $config = $this->configFactory->get('site_health.settings');

      if (!$config->get('enable_monitoring')) {
        return;
      }

      // Rate limiting check
      if (!$this->checkRateLimit()) {
        return;
      }

      $queries = Database::getLog('site_health');
      if (empty($queries)) {
        return;
      }

      // Sampling - only log a percentage of requests
      $sampling_rate = $config->get('sampling_rate') ?: 10; // Default 10%
      if (mt_rand(1, 100) > $sampling_rate) {
        return;
      }

      $current_uri = $this->sanitizeUri(\Drupal::request()->getRequestUri());
      $batch_data = [];

      foreach ($queries as $query) {
        // Skip if query should be excluded
        if ($this->shouldExcludeQuery((string) $query['query'])) {
          continue;
        }

        $sanitized_query = $this->sanitizeQuery((string) $query['query']);
        $batch_data[] = [
          'query_hash' => $this->generateSecureHash($sanitized_query),
          'query' => $sanitized_query,
          'execution_time' => $query['time'] * 1000, // Convert to milliseconds
          'memory_usage' => memory_get_peak_usage(true), // More accurate memory measurement
          'timestamp' => $this->time->getRequestTime(),
          'uri' => $current_uri,
          'caller' => $this->getSecureQueryCaller(),
        ];
      }

      if (!empty($batch_data)) {
        // Use queue for async processing
        $queue = $this->queueFactory->get('site_health_log');
        $queue->createItem(['queries' => $batch_data]);
      }

      // Schedule cleanup if needed
      $this->scheduleCleanupIfNeeded();
    } catch (\Exception $e) {
      $this->loggerFactory->get('site_health')
        ->error('Failed to log page queries: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Process batch logging (called by queue worker).
   */
  public function processBatchLogging(array $batch_data)
  {
    try {
      $query = $this->database->insert('site_health_query_log');
      $query->fields([
        'query_hash',
        'query',
        'execution_time',
        'memory_usage',
        'timestamp',
        'uri',
        'caller'
      ]);

      foreach ($batch_data['queries'] as $data) {
        $query->values($data);
      }

      $query->execute();
    } catch (\Exception $e) {
      $this->loggerFactory->get('site_health')
        ->error('Failed to process batch logging: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Sanitize query to remove sensitive information.
   */
  private function sanitizeQuery($query_string)
  {
    // Remove sensitive data patterns
    foreach ($this->sensitivePatterns as $pattern) {
      $query_string = preg_replace($pattern, 'password = [REDACTED]', $query_string);
    }

    // Remove potential email addresses
    $query_string = preg_replace('/[\w\.-]+@[\w\.-]+\.\w+/', '[EMAIL_REDACTED]', $query_string);

    // Remove potential credit card numbers
    $query_string = preg_replace('/\b\d{4}[\s-]?\d{4}[\s-]?\d{4}[\s-]?\d{4}\b/', '[CARD_REDACTED]', $query_string);

    // Limit query length for storage efficiency
    if (strlen($query_string) > 2000) {
      $query_string = substr($query_string, 0, 2000) . ' [TRUNCATED]';
    }

    return $query_string;
  }

  /**
   * Generate secure hash using SHA-256 instead of MD5.
   */
  private function generateSecureHash($query_string)
  {
    return hash('sha256', $query_string);
  }

  /**
   * Sanitize URI to prevent information leakage.
   */
  private function sanitizeUri($uri)
  {
    // Remove query parameters that might contain sensitive data
    $uri = preg_replace('/[?&](token|key|password|hash|secret)=[^&]*/i', '', $uri);

    // Limit URI length
    if (strlen($uri) > 255) {
      $uri = substr($uri, 0, 255);
    }

    return $uri;
  }

  /**
   * Check if a query should be excluded from logging with improved security.
   */
  private function shouldExcludeQuery($query_string)
  {
    $excluded_tables = $this->getExcludedTables();

    if (empty($excluded_tables)) {
      return FALSE;
    }

    // Normalize query for better matching
    $query_normalized = preg_replace('/\s+/', ' ', strtolower(trim($query_string)));

    foreach ($excluded_tables as $table) {
      $table = trim($table);
      if (!empty($table)) {
        // Use more precise regex matching instead of simple strpos
        $pattern = '/\b' . preg_quote(strtolower($table), '/') . '\b/';
        if (preg_match($pattern, $query_normalized)) {
          return TRUE;
        }
      }
    }

    // Always exclude queries to the query monitor table itself to prevent recursion
    if (preg_match('/\bsite_health_query_log\b/', $query_normalized)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Get the list of excluded tables from configuration.
   */
  private function getExcludedTables()
  {
    $config = $this->configFactory->get('site_health.settings');
    $excluded_tables_string = $config->get('excluded_tables');

    if (empty($excluded_tables_string)) {
      return [];
    }

    // Split by semicolon and clean up whitespace
    $tables = explode(';', $excluded_tables_string);
    return array_map('trim', array_filter($tables));
  }

  /**
   * Get sanitized caller information without exposing sensitive paths.
   */
  private function getSecureQueryCaller()
  {
    $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 15);

    foreach ($backtrace as $trace) {
      if (isset($trace['class']) && isset($trace['function'])) {
        $class = $trace['class'];

        // Only include Drupal core/contrib classes, exclude custom paths
        if (
          strpos($class, 'Drupal\\Core\\') === 0 ||
          strpos($class, 'Drupal\\Component\\') === 0
        ) {
          return basename($class) . '::' . $trace['function'];
        }
      }
    }

    return 'System';
  }

  /**
   * Rate limiting to prevent excessive logging.
   */
  private function checkRateLimit()
  {
    $config = $this->configFactory->get('site_health.settings');
    $rate_limit = $config->get('rate_limit_per_minute') ?: 60; // Default 60 requests per minute

    $key = 'site_health_rate_limit';
    $current_minute = floor($this->time->getRequestTime() / 60);
    $stored_data = $this->state->get($key, ['minute' => 0, 'count' => 0]);

    if ($stored_data['minute'] !== $current_minute) {
      // Reset counter for new minute
      $this->state->set($key, ['minute' => $current_minute, 'count' => 1]);
      return TRUE;
    }

    if ($stored_data['count'] >= $rate_limit) {
      return FALSE;
    }

    $this->state->set($key, [
      'minute' => $current_minute,
      'count' => $stored_data['count'] + 1
    ]);

    return TRUE;
  }

  /**
   * Schedule cleanup if needed.
   */
  private function scheduleCleanupIfNeeded()
  {
    $last_cleanup = $this->state->get('site_health_last_cleanup', 0);
    $cleanup_interval = 3600; // 1 hour

    if (($this->time->getRequestTime() - $last_cleanup) > $cleanup_interval) {
      $queue = $this->queueFactory->get('site_health_cleanup');
      $queue->createItem(['cleanup_time' => $this->time->getRequestTime()]);

      $this->state->set('site_health_last_cleanup', $this->time->getRequestTime());
    }
  }

  /**
   * Clean up old query logs (called by queue worker).
   */
  public function performCleanup()
  {
    try {
      $config = $this->configFactory->get('site_health.settings');
      $retention_days = $config->get('log_retention_days') ?: 7;
      $max_entries = $config->get('max_log_entries') ?: 10000;

      $cutoff_time = $this->time->getRequestTime() - ($retention_days * 24 * 60 * 60);

      // Delete old entries in batches to avoid locking
      $batch_size = 1000;
      do {
          $deleted = $this->database->delete('site_health_query_log')
          ->condition('timestamp', $cutoff_time, '<')
          ->range(0, $batch_size)
          ->execute();
      } while ($deleted == $batch_size);

      // Keep only the most recent entries if we exceed max_entries
      $total_count = $this->database->select('site_health_query_log', 'q')
        ->countQuery()
        ->execute()
        ->fetchField();

      if ($total_count > $max_entries) {
        $subquery = $this->database->select('site_health_query_log', 'q')
          ->fields('q', ['id'])
          ->orderBy('timestamp', 'DESC')
          ->range(0, $max_entries);

        $this->database->delete('site_health_query_log')
          ->condition('id', $subquery, 'NOT IN')
          ->execute();
      }
    } catch (\Exception $e) {
      $this->loggerFactory->get('site_health')
        ->error('Failed to perform cleanup: @message', ['@message' => $e->getMessage()]);
    }
  }
}
