<?php

declare(strict_types=1);

namespace Drupal\site_key_mutator;

use Drupal\Component\Utility\Crypt;
use Drupal\update\UpdateFetcherInterface;
use Symfony\Component\DependencyInjection\Attribute\AsDecorator;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\Attribute\AutowireDecorated;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Decorates the 'update.fetcher' service to mutate the $site_key for privacy.
 */
#[AsDecorator(decorates: 'update.fetcher', priority: 9999, onInvalid: ContainerInterface::EXCEPTION_ON_INVALID_REFERENCE)]
final class UpdateFetcherDecorator implements UpdateFetcherInterface {

  const string CONFIGURE_ALWAYS_EMPTY = 'empty';
  const string CONFIGURE_ALWAYS_RANDOM = 'random';
  const string CONFIGURE_RANDOM_WHEN_NOT_EMPTY = 'random-not-empty';

  const int MUTATE_ALWAYS_EMPTY = 0;
  const int MUTATE_ALWAYS_RANDOM = 1;
  const int MUTATE_RANDOM_IF_EMPTY = 2;

  /**
   * Configured mutation mode.
   *
   * @var int<0,2>
   *
   * @see self::MUTATE_ALWAYS_EMPTY
   * @see self::MUTATE_ALWAYS_RANDOM
   * @see self::MUTATE_RANDOM_IF_EMPTY
   */
  protected int $mutationMode;

  public function __construct(
    #[AutowireDecorated]
    protected UpdateFetcherInterface $inner,
    #[Autowire('%site_key_mutator.mutate_mode%')]
    string $mutation_mode,
  ) {
    $this->mutationMode = match ($mutation_mode) {
      self::CONFIGURE_ALWAYS_EMPTY => self::MUTATE_ALWAYS_EMPTY,
      self::CONFIGURE_ALWAYS_RANDOM => self::MUTATE_ALWAYS_RANDOM,
      self::CONFIGURE_RANDOM_WHEN_NOT_EMPTY => self::MUTATE_RANDOM_IF_EMPTY,
      default => throw new \InvalidArgumentException('Invalid mutation mode provide'),
    };

  }

  /**
   * {@inheritdoc}
   */
  public function getFetchBaseUrl($project): string {
    return $this->inner->getFetchBaseUrl($project);
  }

  /**
   * {@inheritdoc}
   */
  public function fetchProjectData(array $project, $site_key = ''): string {
    return $this->inner->fetchProjectData($project, $this->mutateSiteKey($site_key));
  }

  /**
   * {@inheritdoc}
   */
  public function buildFetchUrl(array $project, $site_key = ''): string {
    return $this->inner->buildFetchUrl($project, $this->mutateSiteKey($site_key));
  }

  /**
   * Mutates the site key if required.
   */
  private function mutateSiteKey(string $site_key = ''): string {

    switch ($this->mutationMode) {
      case self::MUTATE_ALWAYS_EMPTY:
        return '';

      case self::MUTATE_ALWAYS_RANDOM:
        return $this->generateRandomSiteKey();

      case self::MUTATE_RANDOM_IF_EMPTY:
        return $site_key == '' ? '' : $this->generateRandomSiteKey();
    }
  }

  /**
   * Generate a random site key.
   */
  private function generateRandomSiteKey(): string {
    return Crypt::hashBase64(random_bytes(64));
  }

}
