<?php

namespace Drupal\site_settings_extended\Controller;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Drupal\site_settings\SiteSettingsLoaderInterface;
use Drupal\site_settings_extended\SiteSettingsExtendedManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ConfigPagesSettingsController extends ControllerBase {

  protected ?ContentTranslationManagerInterface $contentTranslationManager = NULL;

  protected EntityDisplayRepositoryInterface $entityDisplayRepository;

  protected RendererInterface $renderer;

  protected SiteSettingsExtendedManager $manager;

  protected SiteSettingsLoaderInterface $loader;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->contentTranslationManager = $container->has('content_translation.manager')
      ? $container->get('content_translation.manager') : NULL;
    $instance->languageManager = $container->get('language_manager');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->entityDisplayRepository = $container->get('entity_display.repository');
    $instance->renderer = $container->get('renderer');
    $instance->manager = $container->get('site_settings_extended.manager');
    $instance->loader = $container->get('plugin.manager.site_settings_loader')->getActiveLoaderPlugin();
    return $instance;
  }

  /**
   * Renders the config_pages settings page.
   */
  public function overview() {
    $render = [];
    $group_storage = $this->entityTypeManager()->getStorage('site_setting_group_entity_type');
    $item_storage = $this->entityTypeManager()->getStorage('site_setting_entity');
    $item_view_builder = $this->entityTypeManager()->getViewBuilder('site_setting_entity');
    $current_language = $this->languageManager()->getCurrentLanguage();
    $current_langcode = $current_language->getId();
    $can_add = $this->currentUser()->hasPermission('add site setting entities');

    // Quick link to manage site setting entity types/groups.
    if ($this->currentUser()->hasPermission('administer site setting entities')) {
      $render['_manage'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage site settings'),
        '#url' => Url::fromRoute('entity.site_setting_entity_type.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small', 'button--secondary'],
        ],
      ];
    }

    // Language switcher.
    $languages = $this->languageManager()->getLanguages();
    if (count($languages) > 1) {
      $render['_language'] = [
        '#type' => 'dropbutton',
        '#dropbutton_type' => 'small',
      ];

      // Add the main language first.
      $render['_language']['#links'][$current_language->getId()] = [
        'title' => $current_language->getName(),
        'url' => Url::fromRoute('<nolink>'),
      ];

      // Add any other languages.
      foreach ($languages as $language) {
        if ($language->getId() === $current_langcode) {
          continue;
        }

        $render['_language']['#links'][$language->getId()] = [
          'title' => $language->getName(),
          'url' => Url::fromRoute('entity.site_setting_entity.collection', [], [
            'language' => $language,
          ]),
        ];
      }
    }

    $render['_add'] = [
      '#type' => 'container',
    ];

    if ($can_add) {
      $render['_not_yet_created'] = $this->manager->renderNotYetCreated();
    }

    $render['overview'] = [
      '#theme' => 'admin_page',
      '#blocks' => [],
    ];

    $groups = $group_storage->getQuery()
      ->accessCheck(TRUE)
      ->sort('label')
      ->execute();

    if (empty($groups)) {
      $render['empty'] = [
        '#markup' => $this->t('There are no editable settings yet.'),
      ];
      return $render;
    }

    // Save the available view displays per bundle.
    $view_displays_per_bundle = [];

    // Loop over all the groups and add settings that belong to it.
    foreach ($groups as $group_id) {
      /** @var \Drupal\site_settings\Entity\SiteSettingGroupEntityType $group */
      if (!$group = $group_storage->load($group_id)) {
        continue;
      }

      $items = $this->loader->loadByGroup($group_id);

      if (empty($items)) {
        continue;
      }

      $render['overview']['#blocks'][$group_id] = [
        'title' => $group->label(),
        'content' => [
          '#theme' => 'admin_block_content',
          '#content' => [],
        ],
      ];

      // Add all existing items that belong to this group.
      /** @var \Drupal\site_settings\SiteSettingEntityInterface $item */
      foreach ($items as $item) {
        if (!$item->access('update')) {
          continue;
        }

        $item_id = $item->id();
        /** @var \Drupal\site_settings\SiteSettingEntityTypeInterface $site_setting_type */
        $site_setting_type = $item->get('type')->entity;

        $is_translatable =
          $this->contentTranslationManager?->isEnabled('site_setting_entity', $site_setting_type->id());
        $translation_exists = FALSE;
        if ($is_translatable && $item->hasTranslation($current_langcode)) {
          $item = $item->getTranslation($current_langcode);
          $translation_exists = TRUE;
        }

        // Grab the available view displays for this bundle.
        if (empty($view_displays_per_bundle[$item->bundle()])) {
          $view_displays_per_bundle[$item->bundle()] =
            $this->entityDisplayRepository->getViewModeOptionsByBundle($item->getEntityTypeId(), $item->bundle());
        }

        // Only render the menu description if the 'menu_description' view display exists.
        $menu_description = NULL;
        if (array_key_exists('menu_description', $view_displays_per_bundle[$item->bundle()])) {
          $menu_description = $item_view_builder->view($item, 'menu_description');
          $menu_description = strip_tags($this->renderer->renderInIsolation($menu_description));
        }

        if (!$is_translatable || $translation_exists) {
          $url = $item->toUrl('edit-form');
        }
        else {
          $url = Url::fromRoute('entity.site_setting_entity.content_translation_add', [
            'site_setting_entity' => $item_id,
            'group' => $item->getGroup(),
            'source' => $item->language()->getId(),
            'target' => $current_langcode,
          ]);
        }

        $render['overview']['#blocks'][$group_id]['content']['#content'][$item_id] = [
          'title' => $item->label(),
          'description' => $menu_description,
          'url' => $this->manager->getUrl($url),
        ];

        // Make it clear which translation is being edited.
        if ($is_translatable) {
          if (!$translation_exists) {
            $render['overview']['#blocks'][$group_id]['content']['#content'][$item_id]['title']
              .= ' (' . $this->t('Not translated') . ')';
          }
        }
        else {
          $render['overview']['#blocks'][$group_id]['content']['#content'][$item_id]['title']
            .= ' (' . $this->t('All languages') . ')';
        }

        // For 'multiple'-site setting items we should add a 'create' button on top.
        if ($can_add && $site_setting_type->multiple && !isset($render['_add'][$site_setting_type->id()])) {
          $render['_add'][$site_setting_type->id()] = [
            '#type' => 'link',
            '#title' => $this->t('Add @type', ['@type' => $site_setting_type->label()]),
            '#url' => $this->manager->getUrl(Url::fromRoute('entity.site_setting_entity.add_form', [
              'site_setting_entity_type' => $site_setting_type->id(),
            ])),
            '#attributes' => [
              'class' => ['button', 'button--action', 'button--primary', 'button--small'],
            ],
            '#attached' => [
              'library' => ['core/drupal.dialog.ajax'],
            ],
          ];
        }
      }
    }
    $render['#cache'] = [
      'tags' => ['site_settings_extended'],
    ];
    return $render;
  }

}
