<?php

namespace Drupal\site_settings_extended\Form;

use Drupal\content_translation\ContentTranslationManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Drupal\site_settings\SiteSettingEntityInterface;
use Drupal\site_settings\SiteSettingsLoaderInterface;
use Drupal\site_settings_extended\SiteSettingsExtendedManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

class AllSiteSettingsForm extends FormBase {

  /**
   * Constructs the 'all_site_settings_form' form object.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LanguageManagerInterface $languageManager,
    protected AccountInterface $currentUser,
    protected SiteSettingsExtendedManager $manager,
    protected SiteSettingsLoaderInterface $loader,
    protected ?ContentTranslationManagerInterface $contentTranslationManager,
  ) {}

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('current_user'),
      $container->get('site_settings_extended.manager'),
      $container->get('plugin.manager.site_settings_loader')->getActiveLoaderPlugin(),
      $container->has('content_translation.manager')
        ? $container->get('content_translation.manager') : NULL,
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'all_site_settings_form';
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $group_storage = $this->entityTypeManager->getStorage('site_setting_group_entity_type');
    $item_storage = $this->entityTypeManager->getStorage('site_setting_entity');
    $current_language = $this->languageManager->getCurrentLanguage();
    $current_langcode = $current_language->getId();
    $can_add = $this->currentUser->hasPermission('add site setting entities');
    $can_delete = $this->currentUser->hasPermission('delete site setting entities');

    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'site_settings_extended/admin';

    // Quick link to manage site setting entity types/groups.
    if ($this->currentUser->hasPermission('administer site setting entities')) {
      $form['_manage'] = [
        '#type' => 'link',
        '#title' => $this->t('Manage site settings'),
        '#url' => Url::fromRoute('entity.site_setting_entity_type.collection'),
        '#attributes' => [
          'class' => ['button', 'button--small', 'button--secondary'],
        ],
      ];
    }

    // Language switcher.
    $languages = $this->languageManager->getLanguages();
    if (count($languages) > 1) {
      $form['_language'] = [
        '#type' => 'dropbutton',
        '#dropbutton_type' => 'small',
      ];

      // Add the main language first.
      $form['_language']['#links'][$current_language->getId()] = [
        'title' => $current_language->getName(),
        'url' => Url::fromRoute('<nolink>'),
      ];

      // Add any other languages.
      foreach ($languages as $language) {
        if ($language->getId() === $current_langcode) {
          continue;
        }

        $form['_language']['#links'][$language->getId()] = [
          'title' => $language->getName(),
          'url' => Url::fromRoute('entity.site_setting_entity.collection', [], [
            'language' => $language,
          ]),
        ];
      }
    }

    if ($can_add) {
      $form['_not_yet_created'] = $this->manager->renderNotYetCreated();
    }

    $form['advanced'] = [
      '#type' => 'vertical_tabs',
    ];

    $delta = 0;
    $groups = $group_storage->getQuery()
      ->accessCheck(TRUE)
      ->sort('label')
      ->execute();

    if (empty($groups)) {
      $form['empty'] = [
        '#markup' => $this->t('There are no editable settings yet.'),
      ];
      return $form;
    }

    // Loop over all the groups and add settings that belong to it.
    foreach ($groups as $group_id) {
      /** @var \Drupal\site_settings\Entity\SiteSettingGroupEntityType $group */
      if (!$group = $group_storage->load($group_id)) {
        continue;
      }

      $items = $this->loader->loadByGroup($group_id);
      if (empty($items)) {
        continue;
      }

      $form[$group_id] = [
        '#type' => 'details',
        '#title' => $group->label(),
        '#group' => 'advanced',
        '#site_setting_group_entity_type' => $group_id,
      ];

      $form[$group_id]['_add'] = [
        '#type' => 'item',
      ];

      // Add all existing items that belong to this group.
      /** @var \Drupal\site_settings\SiteSettingEntityInterface $item */
      foreach ($items as $item) {
        if (!$item->access('update')) {
          continue;
        }

        $item_id = $item->id();
        /** @var \Drupal\site_settings\SiteSettingEntityTypeInterface $site_setting_type */
        $site_setting_type = $item->get('type')->entity;

        $is_translatable =
          $this->contentTranslationManager?->isEnabled('site_setting_entity', $site_setting_type->id());
        $translation_exists = FALSE;
        if ($is_translatable && $item->hasTranslation($current_langcode)) {
          $item = $item->getTranslation($current_langcode);
          $translation_exists = TRUE;
        }

        $form[$group_id][$item_id] = [
          '#type' => 'details',
          '#title' => $item->label(),
          '#open' => count($items) === 1,
        ];

        // Make it clear which translation is being edited.
        if ($is_translatable) {
          if (!$translation_exists) {
            $form[$group_id][$item_id]['#title'] .= ' (' . $this->t('Not translated') . ')';
          }
        }
        else {
          $form[$group_id][$item_id]['#title'] .= ' (' . $this->t('All languages') . ')';
        }

        // For 'multiple'-site setting items we should add a 'create' button on top.
        if ($can_add && $site_setting_type->multiple && !isset($form['_add'][$site_setting_type->id()])) {
          $form[$group_id]['_add'][$site_setting_type->id()] = [
            '#type' => 'link',
            '#title' => $this->t('Add @type', ['@type' => $site_setting_type->label()]),
            '#url' => $this->manager->getUrl(Url::fromRoute('entity.site_setting_entity.add_form', [
              'site_setting_entity_type' => $site_setting_type->id(),
            ])),
            '#attributes' => [
              'class' => ['button', 'button--action', 'button--primary'],
            ],
          ];
        }

        $form[$group_id][$item_id]['form'] = [
          '#type' => 'inline_entity_form',
          '#entity_type' => 'site_setting_entity',
          '#bundle' => $item->bundle(),
          '#langcode' => $current_langcode,
          '#default_value' => $item,
          '#op' => 'default',
          '#form_mode' => 'compact',
          '#revision' => FALSE,
          '#save_entity' => FALSE,
          '#ief_row_delta' => $delta,
          '#parents' => [$group_id, $item_id],
          '#ief_labels' => $item->label(),
          '#ief_id' => $delta,
        ];

        if ($is_translatable && !$translation_exists) {
          $form[$group_id][$item_id]['_save_mode'] = [
            '#type' => 'radios',
            '#title' => $this->t('Save mode'),
            '#description' => $this->t('There is no translation yet for this item in the current language.
Please, pick the save mode for this item.'),
            '#options' => [
              'no_save' => $this->t('Do nothing'),
              'create_translation' => $this->t('Create new translation'),
            ],
            '#default_value' => 'no_save',
            '#required' => TRUE,
            '#weight' => -1,
          ];

          // Only show the form if the save mode is correct.
          $form[$group_id][$item_id]['form']['#states'] = [
            'invisible' => [
              ':input[name="' . $group_id . '[' . $item_id . '][_save_mode]"]' => ['value' => 'no_save'],
            ],
          ];
        }

        if ($can_delete) {
          $delete_button_label = $item->isDefaultTranslation()
            ? $this->t('Delete')
            : $this->t('Delete translation');
          $form[$group_id][$item_id]['_delete'] = [
            '#type' => 'link',
            '#title' => $delete_button_label,
            '#url' => $this->manager->getUrl($item->toUrl('delete-form')),
            '#attributes' => [
              'class' => [
                'action-link',
                'action-link--danger',
                'action-link--icon-trash',
              ],
            ],
          ];

          if ($is_translatable && !$translation_exists) {
            // Only show the delete button if the save mode is correct.
            $form[$group_id][$item_id]['_delete']['#states'] = [
              'invisible' => [
                ':input[name="' . $group_id . '[' . $item_id . '][_save_mode]"]' => ['value' => 'no_save'],
              ],
            ];
          }
        }
        $delta++;
      }

      if (Element::children($form[$group_id]) === ['_add']) {
        unset($form[$group_id]);
      }
    }

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
        '#button_type' => 'primary',
        '#disabled' => !$this->currentUser->hasPermission('edit site setting entities'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $this->processEntities(
      $form_state,
      static function (SiteSettingEntityInterface $entity) use ($form_state) {
        $violations = $entity->validate();
        if ($violations->count() > 0) {
          $form_state->setErrorByName(
            $entity->getGroup() . '][' . $entity->id() . '][form',
            implode("\n", $violations->getIterator()->getArrayCopy()),
          );
        }
      }
    );
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->processEntities(
      $form_state,
      static function (SiteSettingEntityInterface $entity) {
        $entity->save();
      }
    );
  }

  /**
   *
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param callable $callback
   *   The callback to run at the end.
   *
   * @return void
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function processEntities(FormStateInterface $form_state, callable $callback): void {
    $item_storage = $this->entityTypeManager->getStorage('site_setting_entity');
    $current_langcode = $this->languageManager->getCurrentLanguage()->getId();

    foreach ($form_state->getValues() as $group => $items) {
      if (!is_array($items)) {
        continue;
      }

      foreach ($items as $entity_id => $entity_values) {
        if (!is_array($entity_values)) {
          continue;
        }

        /** @var \Drupal\site_settings\SiteSettingEntityInterface $entity */
        if (!$entity = $item_storage->load($entity_id)) {
          continue;
        }

        $is_translatable = $this->contentTranslationManager?->isEnabled('site_setting_entity', $entity->bundle());
        if ($is_translatable) {
          $save_mode = $entity_values['_save_mode'] ?? NULL;

          // Save values to the existing translation.
          if ($entity->hasTranslation($current_langcode)) {
            $entity = $entity->getTranslation($current_langcode);
          }
          // Create a new translation if requested.
          elseif ($save_mode === 'create_translation') {
            $entity = $entity->addTranslation($current_langcode, $entity->toArray());
          }
          // Skip this item, no translation is to be made.
          elseif ($save_mode === 'no_save') {
            continue;
          }
        }

        foreach ($entity_values as $field => $value) {
          if ($field === 'langcode' || !$entity->hasField($field)) {
            continue;
          }

          // Media library items.
          if (isset($value['selection'])) {
            $value = $value['selection'];
          }

          $entity->set($field, $value);
        }

        $callback($entity);
      }
    }
  }

}
