<?php

namespace Drupal\site_settings_extended\Plugin\views\area;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area handler to display a 'site_setting_entity' item.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("site_setting_item")]
class SiteSettingItem extends AreaPluginBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['site_setting_type']['default'] = NULL;
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $types = $this->entityTypeManager->getStorage('site_setting_entity_type')->loadMultiple();
    $types = array_map(static function ($type) {
      return $type->label();
    }, $types);

    $form['site_setting_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Site setting type'),
      '#description' => $this->t('Select the site setting type, this should be a NON-MULTIPLE
        type since the first valid item found for the current language will be rendered.'),
      '#default_value' => $this->options['site_setting_type'] ?? NULL,
      '#empty_option' => $this->t('- Select -'),
      '#options' => $types,
      '#required' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    $items = $this->entityTypeManager
      ->getStorage('site_setting_entity')
      ->loadByProperties(['type' => $this->options['site_setting_type']]);

    if (!$item = reset($items)) {
      return [];
    }

    $current_language = $this->languageManager->getCurrentLanguage()->getId();

    /** @var \Drupal\site_settings\SiteSettingEntityInterface $item */
    if ($item->hasTranslation($current_language)) {
      $item = $item->getTranslation($current_language);
    }

    $view_builder = $this->entityTypeManager->getViewBuilder('site_setting_entity');
    $view = $view_builder->view($item);
    return $view_builder->build($view);
  }

}
