<?php

namespace Drupal\site_settings_extended;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;

/**
 * A helper class to renter a table with site setting entities that don't yet exist.
 */
class SiteSettingsExtendedManager {
  use StringTranslationTrait;

  /**
   * Storage for bundle infos.
   *
   * @var array
   */
  protected array $bundles = [];

  protected ?bool $useModals = NULL;

  /**
   * Constructs the helper object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database service.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected Connection $database,
    protected RendererInterface $renderer,
    protected ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Renders the table of site setting entities that don't yet exist.
   */
  public function renderNotYetCreated(): array {
    $this->bundles = $this->entityTypeManager->getStorage('site_setting_entity_type')->loadMultiple();
    $query = $this->database->select('site_setting_entity', 'sse');
    $query->addField('sse', 'type');
    $used_bundles = $query->distinct()->execute()->fetchCol();
    $used_bundles = $used_bundles ?: [];
    $missing_bundles = array_diff(array_keys($this->bundles), $used_bundles);
    $build = [];

    // If we have site settings not yet created.
    if ($missing_bundles) {
      $build['table'] = [
        '#type' => 'table',
        '#header' => [
          'name' => $this->t('Not yet created setting'),
          'operations' => $this->t('Operations'),
        ],
        '#rows' => [],
      ];

      // Sort missing bundles alphabetically by group and label.
      usort($missing_bundles, function ($a, $b) {
        if ($this->bundles[$a]->group == $this->bundles[$b]->group) {
          return ($this->bundles[$a]->label() >= $this->bundles[$b]->label()) ? -1 : 1;
        }
        return $this->bundles[$a]->group >= $this->bundles[$b]->group ? -1 : 1;
      });

      // Boolean to determine whether the 'Settings not yet created' title
      // should be shown.
      foreach ($missing_bundles as $missing) {

        // Settings that have not yet been created rows.
        $url = new Url('entity.site_setting_entity.add_form', [
          'site_setting_entity_type' => $missing,
        ]);
        if ($url->access()) {
          // Add link if user has access.
          $link = [
            '#type' => 'link',
            '#title' => $this->t('Create setting'),
            '#url' => $this->getUrl($url),
            '#attributes' => [
              'class' => ['button', 'button--action'],
            ],
            '#attached' => [
              'library' => ['core/drupal.dialog.ajax'],
            ],
          ];

          array_unshift($build['table']['#rows'], [
            'name' => $this->bundles[$missing]->label(),
            'operations' => $this->renderer->render($link),
          ]);
        }
      }
    }

    return $build;
  }

  /**
   * Retrieves the correct URL.
   */
  public function getUrl(Url $url): Url {
    if (is_null($this->useModals)) {
      $this->useModals = $this->configFactory
        ->get('site_settings_extended.settings')
        ->get('use_modals');
    }

    if ($this->useModals) {
      $url_options = $url->getOptions();
      $url_options['attributes']['class'][] = 'use-ajax';
      $url_options['attributes']['data-dialog-type'] = 'modal';
      $url_options['attributes']['data-dialog-options'] = json_encode([
        'width' => '900',
      ]);
      $url->setOptions($url_options);
    }

    return $url;
  }

}
