<?php

namespace Drupal\site_status_message\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Renderer;
use Drupal\site_status_message\SiteStatusMessageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Site Status Message administration form.
 */
class AdminForm extends ConfigFormBase {

  /**
   * The Module Handler interface.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * Renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected Renderer $renderer;

  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    ModuleHandlerInterface $moduleHandler,
    Renderer $renderer,
    EntityTypeManagerInterface $entity_type_manager
  ) {
    parent::__construct($config_factory, $typed_config_manager);

    $this->moduleHandler = $moduleHandler;
    $this->renderer = $renderer;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): ConfigFormBase {
    /** @var \Drupal\Core\Config\ConfigFactoryInterface $config */
    $config = $container->get('config.factory');

    /** @var \Drupal\Core\Config\TypedConfigManagerInterface $type */
    $type = $container->get('config.typed');

    /** @var \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler */
    $moduleHandler = $container->get('module_handler');

    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = $container->get('renderer');

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager */
    $entityTypeManager = $container->get('entity_type.manager');

    return new static(
      $config,
      $type,
      $moduleHandler,
      $renderer,
      $entityTypeManager
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['site_status_message.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'site_status_message_admin';
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('site_status_message.settings');

    $link_id = $config->get('link');
    $default_node = ($link_id !== NULL && $link_id !== '')
      ? $this->entityTypeManager->getStorage('node')->load($link_id)
      : NULL;

    $form['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Add a message to the top of the site pages.'),
    ];

    // Enable message checkbox.
    $form['enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable message'),
      '#default_value' => $config->get('enable'),
      '#weight' => 0,
    ];

    // Status Message.
    $form['message'] = [
      '#type' => 'textfield',
      '#maxlength' => 256,
      '#size' => 80,
      '#title' => $this->t('Status message'),
      '#default_value' => $config->get('message'),
      '#description' => $this->t('The message to display at the top of each page.'),
      '#placeholder' => $this->t('Enter a message here.'),
      '#weight' => 10,
    ];

    // Show Token browser link if the Token module is installed.
    if ($this->moduleHandler->moduleExists('token')) {
      $token_tree = [
        '#theme' => 'token_tree_link',
        '#token_types' => [],
      ];

      /* @noinspection PhpUnhandledExceptionInspection */
      $rendered_token_tree = $this->renderer->render($token_tree);

      $form['message']['#description'] .= ' ' . $this->t(
        'This field supports tokens. @browse_tokens_link', [
          '@browse_tokens_link' => $rendered_token_tree,
        ]);
    }

    // Show link checkbox.
    $form['showlink'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Read more page'),
      '#default_value' => $config->get('showlink'),
      '#description' => $this->t('Optional "Read More" link to provide the viewer with more information.'),
      '#weight' => 20,
    ];

    // Internal page link.
    $form['link'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'node',
      '#title' => $this->t('More information page'),
      '#size' => '80',
      '#maxlength' => 256,
      '#default_value' => $default_node,
      '#description' => $this->t('An optional internal link to show more information about the status message.'),
      '#weight' => 30,
      '#states' => [
        'visible' => [
          ':input[name="showlink"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // The "Read More" page link text.
    $form['readmore'] = [
      '#type' => 'textfield',
      '#size' => 80,
      '#maxlength' => 128,
      '#title' => $this->t('More information link text'),
      '#default_value' => $config->get('readmore'),
      '#description' => $this->t('The text to display on the "More Information" link.'),
      '#placeholder' => $this->t('Enter the read more text here.'),
      '#weight' => 40,
      '#states' => [
        'visible' => [
          ':input[name="showlink"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // The pages on the site where the message should be displayed.
    $form['display'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Display options'),
      '#weight' => 50,
    ];

    $form['display']['display_options'] = [
      '#type' => 'radios',
      '#title' => $this->t('Where on the site should the message be displayed.'),
      '#options' => $this->getDisplayOptions(),
      '#default_value' => $config->get('display_options'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Save the config with all data.
    $this->config('site_status_message.settings')
      ->set('enable', (bool) $form_state->getValue('enable'))
      ->set('message', $form_state->getValue('message'))
      ->set('showlink', (bool) $form_state->getValue('showlink'))
      ->set('link', $form_state->getValue('link') ?: NULL)
      ->set('readmore', $form_state->getValue('readmore'))
      ->set('display_options', $form_state->getValue('display_options'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Display options where to show the message.
   *
   * @return array
   *   Array of options where the message can be displayed.
   */
  private function getDisplayOptions(): array {
    return [
      SiteStatusMessageInterface::SITE_STATUS_MESSAGE_PUBLIC => $this->t('Public-facing site pages'),
      SiteStatusMessageInterface::SITE_STATUS_MESSAGE_ADMIN => $this->t('Admin pages'),
      SiteStatusMessageInterface::SITE_STATUS_MESSAGE_BOTH => $this->t('Both public-facing site and admin pages'),
    ];
  }

}
