<?php

declare(strict_types=1);

namespace Drupal\RecipeKit\SiteTemplate\Command;

use Composer\Command\BaseCommand;
use Composer\Composer;
use Composer\Console\Input\InputOption;
use Drupal\Component\Serialization\Json;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Recipe\Recipe;
use Drupal\RecipeKit\SiteTemplate\Plugin;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Process\PhpExecutableFinder;
use Symfony\Component\Process\Process;

final class AddExtensionCommand extends BaseCommand {

  public function __construct(private readonly Composer $composer) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  protected function configure(): void {
    $this->setName('site-template:extension:add')
      ->setAliases(['st:extadd'])
      ->setDescription('Adds an already-installed module or recipe constraint into a site template.')
      ->addArgument('template', InputArgument::REQUIRED, 'The name of the site template to modify.')
      ->addArgument('package', InputArgument::REQUIRED, 'The name of the Composer package to add to the site template.')
      ->addOption('add', 'a', InputOption::VALUE_NEGATABLE, "Add the package to the recipe's install list.", TRUE);
  }

  /**
   * {@inheritdoc}
   */
  protected function execute(InputInterface $input, OutputInterface $output): int {
    $prefix = [
      (new PhpExecutableFinder())->find(),
      $_ENV['COMPOSER_BINARY'],
    ];
    $package_name = $input->getArgument('package');

    // First, get info about the package.
    $output->writeln('Getting package information...');
    $process = new Process([
      ...$prefix,
      'info',
      $package_name,
      '--format=json',
      '--available',
    ]);
    $info = Json::decode($process->mustRun()->getOutput());

    // Add the package as a dependency of the site template.
    $output->writeln('Adding package to site template dependencies...');
    $working_dir = implode(DIRECTORY_SEPARATOR, [
      Plugin::getRecipeDirectory(),
      $input->getArgument('template'),
    ]);
    $process = new Process([
      ...$prefix,
      'require',
      "$package_name:$constraint",
      '--no-update',
      "--working-dir=$working_dir",
    ]);
    $process->mustRun();

    if ($input->getOption('add')) {
      $output->writeln('Adding to recipes list...');
      $file = implode(DIRECTORY_SEPARATOR, [$working_dir, 'recipe.yml']);
      $data = file_get_contents($file);
      $data = Yaml::decode($data);

      if ($info['type'] === Recipe::COMPOSER_PROJECT_TYPE) {
        $data['recipes'][] = $package_name;
      }
      else {
        $data['install'][] = $package_name;
      }
      file_put_contents($file, Yaml::encode($data));
    }

    return 0;
  }

}
