<?php

declare(strict_types=1);

namespace Drupal\siteimprove_accessibility;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\siteimprove_accessibility\Repository\AlfaScanRepository;
use Drupal\siteimprove_accessibility\Repository\DailyStatsRepository;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Handles the daily aggregation cron job for Siteimprove Accessibility.
 */
class DailyStatsAggregationCron {

  /**
   * Daily stats repository.
   *
   * @var \Drupal\siteimprove_accessibility\Repository\DailyStatsRepository
   */
  protected DailyStatsRepository $dailyStatsRepository;

  /**
   * Daily stats processor.
   *
   * @var \Drupal\siteimprove_accessibility\DailyStatsProcessor
   */
  protected DailyStatsProcessor $dailyStatsProcessor;

  /**
   * State service for tracking last cron execution.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * Logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Time service for getting timestamps.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * AlfaScanRepository service.
   *
   * @var \Drupal\siteimprove_accessibility\Repository\AlfaScanRepository
   */
  protected AlfaScanRepository $alfaScanRepository;

  /**
   * Constructs a new DailyStatServiceAggregationCron.
   *
   * @param \Drupal\siteimprove_accessibility\Repository\DailyStatsRepository $daily_stats_repository
   *   The daily stats repository.
   * @param \Drupal\siteimprove_accessibility\DailyStatsProcessor $daily_stats_processor
   *   The daily stats processor.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\siteimprove_accessibility\Repository\AlfaScanRepository $alfa_scan_repository
   *   AlfaScanRepository service.
   */
  public function __construct(
    DailyStatsRepository $daily_stats_repository,
    DailyStatsProcessor $daily_stats_processor,
    StateInterface $state,
    LoggerChannelFactoryInterface $logger_factory,
    TimeInterface $time,
    AlfaScanRepository $alfa_scan_repository,
  ) {
    $this->dailyStatsRepository = $daily_stats_repository;
    $this->dailyStatsProcessor = $daily_stats_processor;
    $this->state = $state;
    $this->logger = $logger_factory->get('siteimprove_accessibility');
    $this->time = $time;
    $this->alfaScanRepository = $alfa_scan_repository;
  }

  /**
   * Runs the aggregation of daily stats.
   */
  public function aggregateDailyStats(bool $manual = FALSE): void {
    $currentTime = $this->time->getRequestTime();
    $lastRun = $this->state->get('siteimprove_accessibility.last_cron_run', 0);
    if ($manual) {
      $lastRun = 0;
    }

    $midnightToday = strtotime('midnight');
    $yesterday = strtotime('midnight -1 day', $currentTime);

    if ($lastRun < $midnightToday) {
      $this->logger->notice('Running daily cron tasks for Siteimprove Accessibility module.');

      $this->alfaScanRepository->cleanUp();

      $aggregatedStats = $this->dailyStatsProcessor->getAggregatedIssues();

      $this->dailyStatsRepository->createOrUpdateStats(
        $yesterday,
        $aggregatedStats
      );

      $this->state->set('siteimprove_accessibility.last_cron_run', $currentTime);
    }
  }

}
