<?php

declare(strict_types=1);

namespace Drupal\siteimprove_accessibility\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\siteimprove_accessibility\AlfaScanPreprocess;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Siteimprove Alfa settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The AlfaScanPreprocess service.
   *
   * @var \Drupal\siteimprove_accessibility\AlfaScanPreprocess
   */
  private AlfaScanPreprocess $alfaScanPreprocess;

  /**
   * Constructs a new SettingsForm.
   */
  public function __construct(AlfaScanPreprocess $alfa_scan_preprocess) {
    $this->alfaScanPreprocess = $alfa_scan_preprocess;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('siteimprove_accessibility.alfa_scan_preprocess')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'siteimprove_accessibility_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['siteimprove_accessibility.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('siteimprove_accessibility.settings');

    $form['enable_manual_scans'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable widget.'),
      '#description' => $this->t('Display widget when previewing pages.'),
      '#default_value' => $config->get('enable_manual_scans') ?? FALSE,
    ];

    $form['preview_auto_scan'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Accessibility auto-check'),
      '#description' => $this->t('Enable accessibility auto-check on page previews.'),
      '#default_value' => $config->get('preview_auto_scan') ?? FALSE,
    ];

    $form['pendo_tracking'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Usage tracking'),
      '#description' => $this->t('Enable anonymous usage data collection.'),
      '#default_value' => $config->get('pendo_tracking') ?? FALSE,
      '#attributes' => [
        'data-observe-key' => 'a11y-Drupal-UsageTrackingCheckbox',
      ],
    ];

    // Generate a link to the permissions page.
    $permissions_url = Url::fromRoute('user.admin_permissions');
    $permissions_link = Link::fromTextAndUrl($this->t('Manage Permissions'), $permissions_url)
      ->toString();
    $support_url = Url::fromUri('https://help.siteimprove.com/support/home', ['attributes' => ['target' => '_blank']]);
    $support_link = Link::fromTextAndUrl($this->t('Get help'), $support_url)
      ->toString();

    // Add the link to the form.
    $form['permissions_link'] = [
      '#markup' => '<p>' . $this->t('You can manage permissions here: @link', ['@link' => $permissions_link]) . '</p>',
    ];

    $form['support_link'] = [
      '#markup' => '<p>' . $this->t("Need assistance? @link", ['@link' => $support_link]) . '</p>',
    ];

    $form['terms_conditions'] = [
      '#type' => 'link',
      '#title' => $this->t('Terms and Conditions'),
      '#url' => Url::fromUri('https://www.siteimprove.com/legal/accessibility-plugin-terms-of-use/',
        ['attributes' => ['target' => '_blank']]),
      '#attributes' => ['class' => ['terms-link']],
    ];

    $this->alfaScanPreprocess->attachPendo($form);

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('siteimprove_accessibility.settings')
      ->set('preview_auto_scan', $form_state->getValue('preview_auto_scan'))
      ->set('enable_manual_scans', $form_state->getValue('enable_manual_scans'))
      ->set('pendo_tracking', $form_state->getValue('pendo_tracking'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
