<?php

declare(strict_types=1);

namespace Drupal\siteimprove_accessibility\Plugin\rest\resource;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\rest\Attribute\RestResource;
use Drupal\rest\ModifiedResourceResponse;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\siteimprove_accessibility\Repository\AlfaScanRepository;
use Drupal\siteimprove_accessibility\Repository\OccurrenceRepository;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;

#[RestResource(
  id: 'siteimprove_accessibility_alfa_scan_resource',
  label: new TranslatableMarkup('Alfa Scan'),
  uri_paths: [
    'create' => '/siteimprove-accessibility/save-scan',
  ],
)]
/**
 * Provides a REST method to create AlfaScan record.
 */
final class AlfaScanResource extends ResourceBase {

  /**
   * AlfaScanRepository service.
   *
   * @var \Drupal\siteimprove_accessibility\Repository\AlfaScanRepository
   */
  protected AlfaScanRepository $alfaScanRepository;

  /**
   * OccurrenceRepository service.
   *
   * @var \Drupal\siteimprove_accessibility\Repository\OccurrenceRepository
   */
  protected OccurrenceRepository $occurrenceRepository;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a new AlfaScanResource.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    AlfaScanRepository $alfa_scan_repository,
    OccurrenceRepository $occurrence_repository,
    AccountProxyInterface $current_user,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->alfaScanRepository = $alfa_scan_repository;
    $this->occurrenceRepository = $occurrence_repository;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('siteimprove_accessibility.alfa_scan_repository'),
      $container->get('siteimprove_accessibility.occurrence_repository'),
      $container->get('current_user'),
    );
  }

  /**
   * Handles POST requests to save a new AlfaScan record.
   */
  public function post(Request $request): ModifiedResourceResponse {
    if (!$this->currentUser->hasPermission('run siteimprove_accessibility scan')) {
      throw new AccessDeniedHttpException('You do not have permission to access this resource.');
    }

    $data = json_decode($request->getContent(), TRUE);
    $nuuid = $data['scannedNode']['uuid'] ? (string) $data['scannedNode']['uuid'] : NULL;
    $scan_results = isset($data['scan_results']) ? (array) $data['scan_results'] : NULL;
    $scan_stats = isset($data['scan_stats']) ? (array) $data['scan_stats'] : NULL;

    if (!$nuuid || !isset($scan_results) || !isset($scan_stats)) {
      throw new BadRequestHttpException('Invalid request: Missing required fields.');
    }

    $scan = NULL;

    try {
      $scan = $this->alfaScanRepository->createUpdateScan(
        $nuuid,
        (array) $data['scan_results']
      );
    }
    catch (\Exception $exception) {
      $this->logger->error($exception->getMessage());
    }
    if (!$scan) {
      return new ModifiedResourceResponse(['message' => 'Scan was not saved'], 500);
    }

    $this->occurrenceRepository->deleteOccurrences($scan);

    foreach ($scan_stats as $rule => $details) {
      $rule = $this->occurrenceRepository->createUpdateRule($rule, $details['conformance']);
      $this->occurrenceRepository->createOccurrence($scan, $rule, $details['occurrence']);
    }

    return new ModifiedResourceResponse($this->createResponseSummary($scan_stats), 200);
  }

  /**
   * Creates summary on AlfaScan save.
   *
   * @param array $scan_stats
   *   Scan results.
   *
   * @return array
   *   Filtered Scan results.
   */
  protected function createResponseSummary(array $scan_stats): array {
    $scan_stats = array_filter(
      $scan_stats,
      function ($item) {
        return !empty($item);
      }
    );

    return [
      'count_issues' => count($scan_stats),
    ];
  }

}
