/**
 * @file
 * JavaScript for site content override forms.
 */
/* jshint camelcase: false */
/* global Drupal, drupalSettings, once */

(function (once, Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.sitesGroupOverrides = {
    attach: function (context) {
      const conf = drupalSettings && drupalSettings.sitesGroupOverrides ? drupalSettings.sitesGroupOverrides : {};

      // Define helper before use to avoid linter warnings.
      function setLabel(fieldName) {
        const cfg = conf[fieldName];
        if (!cfg) { return; }
        const labelEl = document.querySelector(cfg.label_selector);
        const inputs = document.querySelectorAll(cfg.value_selector);
        if (!labelEl || inputs.length === 0) { return; }

        // Determine current value. For multiple inputs (e.g., radios), prefer the checked one.
        let currentVal = '';
        if (inputs.length > 1 && (inputs[0].type === 'radio' || inputs[0].type === 'checkbox')) {
          const checked = Array.prototype.find.call(inputs, function (i) { return i.checked; });
          currentVal = checked ? checked.value : '';
        }
        else {
          currentVal = inputs[0].value;
        }

        const isOriginal = (currentVal === cfg.original_value);
        const text = isOriginal ? Drupal.t('Original value') : Drupal.t('Overridden');
        labelEl.textContent = '(' + text + ')';
      }

      Object.keys(conf).forEach(function (fieldName) {
        const valueSelector = conf[fieldName].value_selector;
        // Bind once per element in the current context.
        once('sites-group-overrides-change', valueSelector, context).forEach(function (el) {
          // Trigger on both input and change to catch more cases.
          el.addEventListener('input', function () { setLabel(fieldName); });
          el.addEventListener('change', function () { setLabel(fieldName); });
        });
        // Initialize label on attach.
        setLabel(fieldName);
      });

      // React on CKEditor 5 changes: when present, update source and trigger change.
      // Delay a bit to ensure editors are initialized.
      setTimeout(function () {
        Object.keys(conf).forEach(function (fieldName) {
          const selector = conf[fieldName].value_selector;
          once('sites-group-overrides-ckeditor', selector, context).forEach(function (el) {
            const id = el.getAttribute('data-ckeditor5-id');
            if (id !== null && Drupal.CKEditor5Instances && typeof Drupal.CKEditor5Instances.get === 'function') {
              const editor = Drupal.CKEditor5Instances.get(id);
              if (editor && editor.model && editor.model.document) {
                editor.model.document.on('change:data', function () {
                  editor.updateSourceElement();
                  if (editor.sourceElement) {
                    editor.sourceElement.dispatchEvent(new Event('input', { bubbles: true }));
                    editor.sourceElement.dispatchEvent(new Event('change', { bubbles: true }));
                  }
                });
              }
            }
          });
        });
      }, 500);
    }
  };

})(once, Drupal, drupalSettings);
